<?php
/**
 * Gerenciamento de Usuários Administrativos
 */

require_once '../config/config.php';
require_once 'auth.php';

// Protege a página
requireAdminAuth();

$user = getAdminUser();
$db = Database::getInstance()->getConnection();

// Processar ações
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

// Deletar usuário
if ($action === 'delete' && isset($_GET['id'])) {
    try {
        // Não permitir deletar o próprio usuário
        if ($_GET['id'] == $user['id']) {
            $error = "Você não pode deletar sua própria conta!";
        } else {
            $stmt = $db->prepare("DELETE FROM admin_users WHERE id = ?");
            $stmt->execute([$_GET['id']]);
            $message = "Usuário deletado com sucesso!";
        }
        $action = 'list';
    } catch (Exception $e) {
        $error = "Erro ao deletar: " . $e->getMessage();
    }
}

// Salvar usuário (criar/editar)
if ($_POST && isset($_POST['save_user'])) {
    try {
        $name = $_POST['name'];
        $email = $_POST['email'];
        $role = $_POST['role'];
        $active = isset($_POST['active']) ? 1 : 0;
        
        if (isset($_POST['id']) && $_POST['id']) {
            // Editar
            $id = $_POST['id'];
            
            // Verificar se não está tentando desativar a própria conta
            if ($id == $user['id'] && !$active) {
                $error = "Você não pode desativar sua própria conta!";
            } else {
                if (!empty($_POST['password'])) {
                    // Atualizar com nova senha
                    $password = password_hash($_POST['password'], PASSWORD_DEFAULT);
                    $stmt = $db->prepare("UPDATE admin_users SET name=?, email=?, password=?, role=?, active=? WHERE id=?");
                    $stmt->execute([$name, $email, $password, $role, $active, $id]);
                } else {
                    // Atualizar sem alterar senha
                    $stmt = $db->prepare("UPDATE admin_users SET name=?, email=?, role=?, active=? WHERE id=?");
                    $stmt->execute([$name, $email, $role, $active, $id]);
                }
                $message = "Usuário atualizado com sucesso!";
            }
        } else {
            // Criar
            if (empty($_POST['password'])) {
                $error = "Senha é obrigatória para novos usuários!";
            } else {
                // Verificar se email já existe
                $stmt = $db->prepare("SELECT id FROM admin_users WHERE email = ?");
                $stmt->execute([$email]);
                if ($stmt->fetch()) {
                    $error = "Este email já está em uso!";
                } else {
                    $password = password_hash($_POST['password'], PASSWORD_DEFAULT);
                    $stmt = $db->prepare("INSERT INTO admin_users (name, email, password, role, active) VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([$name, $email, $password, $role, $active]);
                    $message = "Usuário criado com sucesso!";
                }
            }
        }
        
        if (!$error) {
            $action = 'list';
        }
    } catch (Exception $e) {
        $error = "Erro ao salvar: " . $e->getMessage();
    }
}

// Buscar usuário para edição
$edit_user = null;
if ($action === 'edit' && isset($_GET['id'])) {
    try {
        $stmt = $db->prepare("SELECT * FROM admin_users WHERE id = ?");
        $stmt->execute([$_GET['id']]);
        $edit_user = $stmt->fetch();
        if (!$edit_user) {
            $error = "Usuário não encontrado!";
            $action = 'list';
        }
    } catch (Exception $e) {
        $error = "Erro ao carregar usuário: " . $e->getMessage();
        $action = 'list';
    }
}

// Listar usuários
$users = [];
if ($action === 'list') {
    try {
        $stmt = $db->query("SELECT * FROM admin_users ORDER BY name");
        $users = $stmt->fetchAll();
    } catch (Exception $e) {
        $error = "Erro ao carregar usuários: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Usuários - Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-100">
    <!-- Header -->
    <header class="bg-blue-900 text-white shadow-lg">
        <div class="container mx-auto px-4 py-3">
            <div class="flex justify-between items-center">
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-white hover:text-gray-300">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Voltar ao Dashboard
                    </a>
                    <h1 class="text-xl font-bold">Gerenciar Usuários</h1>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-sm">
                        <i class="fas fa-user mr-1"></i>
                        <?php echo htmlspecialchars($user['name']); ?>
                    </span>
                    <a href="?action=logout" class="bg-red-600 hover:bg-red-700 px-3 py-1 rounded text-sm">
                        <i class="fas fa-sign-out-alt mr-1"></i>
                        Sair
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="container mx-auto px-4 py-6">
        <!-- Mensagens -->
        <?php if ($message): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <!-- Formulário de Criação/Edição -->
            <div class="lg:col-span-1">
                <div class="bg-white rounded-lg shadow">
                    <div class="p-6 border-b">
                        <h2 class="text-xl font-semibold">
                            <?php echo $action === 'edit' ? 'Editar Usuário' : 'Novo Usuário'; ?>
                        </h2>
                    </div>
                    <form method="POST" class="p-6 space-y-4">
                        <?php if ($edit_user): ?>
                            <input type="hidden" name="id" value="<?php echo $edit_user['id']; ?>">
                        <?php endif; ?>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Nome</label>
                            <input type="text" name="name" required
                                   value="<?php echo htmlspecialchars($edit_user['name'] ?? ''); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Email</label>
                            <input type="email" name="email" required
                                   value="<?php echo htmlspecialchars($edit_user['email'] ?? ''); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Senha</label>
                            <input type="password" name="password" 
                                   <?php echo $action !== 'edit' ? 'required' : ''; ?>
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <?php if ($action === 'edit'): ?>
                                <p class="text-xs text-gray-500 mt-1">Deixe em branco para manter a senha atual</p>
                            <?php endif; ?>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Função</label>
                            <select name="role" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="admin" <?php echo ($edit_user['role'] ?? '') === 'admin' ? 'selected' : ''; ?>>Administrador</option>
                                <option value="editor" <?php echo ($edit_user['role'] ?? '') === 'editor' ? 'selected' : ''; ?>>Editor</option>
                                <option value="author" <?php echo ($edit_user['role'] ?? '') === 'author' ? 'selected' : ''; ?>>Autor</option>
                            </select>
                        </div>

                        <div class="flex items-center">
                            <input type="checkbox" name="active" id="active" value="1" 
                                   <?php echo ($edit_user['active'] ?? 1) ? 'checked' : ''; ?>
                                   class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                            <label for="active" class="ml-2 block text-sm text-gray-900">
                                Usuário ativo
                            </label>
                        </div>

                        <div class="flex justify-between">
                            <?php if ($action === 'edit'): ?>
                                <a href="?action=list" class="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded">
                                    Cancelar
                                </a>
                            <?php else: ?>
                                <div></div>
                            <?php endif; ?>
                            <button type="submit" name="save_user" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded">
                                <i class="fas fa-save mr-2"></i>
                                <?php echo $action === 'edit' ? 'Atualizar' : 'Criar'; ?>
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Lista de Usuários -->
            <div class="lg:col-span-2">
                <div class="bg-white rounded-lg shadow">
                    <div class="p-6 border-b flex justify-between items-center">
                        <h2 class="text-xl font-semibold">Usuários do Sistema</h2>
                        <a href="?action=new" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded text-sm">
                            <i class="fas fa-plus mr-2"></i>
                            Novo Usuário
                        </a>
                    </div>
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Usuário</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Função</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Último Login</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ações</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php if (empty($users)): ?>
                                    <tr>
                                        <td colspan="5" class="px-6 py-4 text-center text-gray-500">
                                            Nenhum usuário encontrado.
                                        </td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($users as $u): ?>
                                        <tr>
                                            <td class="px-6 py-4">
                                                <div>
                                                    <div class="font-medium text-gray-900">
                                                        <?php echo htmlspecialchars($u['name']); ?>
                                                        <?php if ($u['id'] == $user['id']): ?>
                                                            <span class="text-xs text-blue-600">(Você)</span>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="text-sm text-gray-500">
                                                        <?php echo htmlspecialchars($u['email']); ?>
                                                    </div>
                                                </div>
                                            </td>
                                            <td class="px-6 py-4">
                                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                                    <?php 
                                                    switch($u['role']) {
                                                        case 'admin': echo 'bg-red-100 text-red-800'; break;
                                                        case 'editor': echo 'bg-yellow-100 text-yellow-800'; break;
                                                        case 'author': echo 'bg-green-100 text-green-800'; break;
                                                        default: echo 'bg-gray-100 text-gray-800';
                                                    }
                                                    ?>">
                                                    <?php 
                                                    switch($u['role']) {
                                                        case 'admin': echo 'Administrador'; break;
                                                        case 'editor': echo 'Editor'; break;
                                                        case 'author': echo 'Autor'; break;
                                                        default: echo ucfirst($u['role']);
                                                    }
                                                    ?>
                                                </span>
                                            </td>
                                            <td class="px-6 py-4">
                                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                                    <?php echo $u['active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                                    <?php echo $u['active'] ? 'Ativo' : 'Inativo'; ?>
                                                </span>
                                            </td>
                                            <td class="px-6 py-4 text-sm text-gray-900">
                                                <?php echo $u['last_login'] ? date('d/m/Y H:i', strtotime($u['last_login'])) : 'Nunca'; ?>
                                            </td>
                                            <td class="px-6 py-4 text-sm space-x-2">
                                                <a href="?action=edit&id=<?php echo $u['id']; ?>" 
                                                   class="text-blue-600 hover:text-blue-900">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                                <?php if ($u['id'] != $user['id']): ?>
                                                    <a href="?action=delete&id=<?php echo $u['id']; ?>" 
                                                       class="text-red-600 hover:text-red-900"
                                                       onclick="return confirm('Tem certeza que deseja deletar este usuário?')">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                <?php else: ?>
                                                    <span class="text-gray-400" title="Você não pode deletar sua própria conta">
                                                        <i class="fas fa-trash"></i>
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
