<?php
/**
 * Gerenciamento de Notícias/Posts
 */

require_once __DIR__ . '/../../config/config.php';
require_once 'auth.php';

// Protege a página
requireAdminAuth();

$user = getAdminUser();
$db = Database::getInstance()->getConnection();

// Processar ações
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

// Deletar post
if ($action === 'delete' && isset($_GET['id'])) {
    try {
        $stmt = $db->prepare("DELETE FROM posts WHERE id = ?");
        $stmt->execute([$_GET['id']]);
        $message = "Notícia deletada com sucesso!";
        $action = 'list';
    } catch (Exception $e) {
        $error = "Erro ao deletar: " . $e->getMessage();
    }
}

// Salvar post (criar/editar)
if ($_POST && isset($_POST['save_post'])) {
    try {
        $title = $_POST['title'];
        $content = $_POST['content'];
        $excerpt = $_POST['excerpt'];
        $category_id = $_POST['category_id'] ?: null;
        $status = $_POST['status'];
        $featured = isset($_POST['featured']) ? 1 : 0;
        $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title)));
        
        if (isset($_POST['id']) && $_POST['id']) {
            // Editar
            $stmt = $db->prepare("UPDATE posts SET title=?, content=?, excerpt=?, category_id=?, status=?, featured=?, slug=?, updated_at=NOW() WHERE id=?");
            $stmt->execute([$title, $content, $excerpt, $category_id, $status, $featured, $slug, $_POST['id']]);
            $message = "Notícia atualizada com sucesso!";
        } else {
            // Criar
            $stmt = $db->prepare("INSERT INTO posts (title, content, excerpt, category_id, status, featured, slug, author_id, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())");
            $stmt->execute([$title, $content, $excerpt, $category_id, $status, $featured, $slug, $user['id']]);
            $message = "Notícia criada com sucesso!";
        }
        $action = 'list';
    } catch (Exception $e) {
        $error = "Erro ao salvar: " . $e->getMessage();
    }
}

// Buscar categorias para o select
$categories = [];
try {
    $stmt = $db->query("SELECT id, name FROM categories ORDER BY name");
    $categories = $stmt->fetchAll();
} catch (Exception $e) {
    // Ignorar erro se tabela não existir
}

// Buscar post para edição
$post = null;
if ($action === 'edit' && isset($_GET['id'])) {
    try {
        $stmt = $db->prepare("SELECT * FROM posts WHERE id = ?");
        $stmt->execute([$_GET['id']]);
        $post = $stmt->fetch();
        if (!$post) {
            $error = "Notícia não encontrada!";
            $action = 'list';
        }
    } catch (Exception $e) {
        $error = "Erro ao carregar notícia: " . $e->getMessage();
        $action = 'list';
    }
}

// Listar posts
$posts = [];
if ($action === 'list') {
    try {
        $stmt = $db->query("
            SELECT p.*, c.name as category_name, u.full_name as author_name 
            FROM posts p 
            LEFT JOIN categories c ON p.category_id = c.id 
            LEFT JOIN users u ON p.author_id = u.id 
            ORDER BY p.created_at DESC
        ");
        $posts = $stmt->fetchAll();
    } catch (Exception $e) {
        $error = "Erro ao carregar notícias: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Notícias - Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script src="https://cdn.ckeditor.com/4.16.2/standard/ckeditor.js"></script>
</head>
<body class="bg-gray-100">
    <!-- Header -->
    <header class="bg-blue-900 text-white shadow-lg">
        <div class="container mx-auto px-4 py-3">
            <div class="flex justify-between items-center">
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-white hover:text-gray-300">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Voltar ao Dashboard
                    </a>
                    <h1 class="text-xl font-bold">Gerenciar Notícias</h1>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-sm">
                        <i class="fas fa-user mr-1"></i>
                        <?php echo htmlspecialchars($user['name']); ?>
                    </span>
                    <a href="?action=logout" class="bg-red-600 hover:bg-red-700 px-3 py-1 rounded text-sm">
                        <i class="fas fa-sign-out-alt mr-1"></i>
                        Sair
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="container mx-auto px-4 py-6">
        <!-- Mensagens -->
        <?php if ($message): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($action === 'list'): ?>
            <!-- Lista de Posts -->
            <div class="bg-white rounded-lg shadow">
                <div class="p-6 border-b flex justify-between items-center">
                    <h2 class="text-xl font-semibold">Notícias</h2>
                    <a href="?action=new" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded">
                        <i class="fas fa-plus mr-2"></i>
                        Nova Notícia
                    </a>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Título</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Categoria</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Data</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ações</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            <?php foreach ($posts as $post): ?>
                                <tr>
                                    <td class="px-6 py-4">
                                        <div>
                                            <div class="font-medium text-gray-900">
                                                <?php echo htmlspecialchars($post['title']); ?>
                                                <?php if ($post['featured']): ?>
                                                    <span class="ml-2 inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                                                        Destaque
                                                    </span>
                                                <?php endif; ?>
                                            </div>
                                            <div class="text-sm text-gray-500">
                                                Por: <?php echo htmlspecialchars($post['author_name'] ?? 'Desconhecido'); ?>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <?php echo htmlspecialchars($post['category_name'] ?? 'Sem categoria'); ?>
                                    </td>
                                    <td class="px-6 py-4">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                            <?php echo $post['status'] === 'published' ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'; ?>">
                                            <?php echo $post['status'] === 'published' ? 'Publicado' : 'Rascunho'; ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-500">
                                        <?php echo date('d/m/Y H:i', strtotime($post['created_at'])); ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm space-x-2">
                                        <a href="?action=edit&id=<?php echo $post['id']; ?>" 
                                           class="text-blue-600 hover:text-blue-900">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <a href="?action=delete&id=<?php echo $post['id']; ?>" 
                                           class="text-red-600 hover:text-red-900"
                                           onclick="return confirm('Tem certeza que deseja deletar esta notícia?')">
                                            <i class="fas fa-trash"></i>
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

        <?php else: ?>
            <!-- Formulário de Criação/Edição -->
            <div class="bg-white rounded-lg shadow">
                <div class="p-6 border-b">
                    <h2 class="text-xl font-semibold">
                        <?php echo $action === 'edit' ? 'Editar Notícia' : 'Nova Notícia'; ?>
                    </h2>
                </div>
                <form method="POST" class="p-6 space-y-6">
                    <?php if ($post): ?>
                        <input type="hidden" name="id" value="<?php echo $post['id']; ?>">
                    <?php endif; ?>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Título</label>
                        <input type="text" name="title" required
                               value="<?php echo htmlspecialchars($post['title'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Resumo</label>
                        <textarea name="excerpt" rows="3"
                                  class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"><?php echo htmlspecialchars($post['excerpt'] ?? ''); ?></textarea>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Conteúdo</label>
                        <textarea name="content" id="content" rows="10"
                                  class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"><?php echo htmlspecialchars($post['content'] ?? ''); ?></textarea>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Categoria</label>
                            <select name="category_id" 
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="">Sem categoria</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>"
                                            <?php echo ($post['category_id'] ?? '') == $category['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($category['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                            <select name="status" 
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="draft" <?php echo ($post['status'] ?? '') === 'draft' ? 'selected' : ''; ?>>Rascunho</option>
                                <option value="published" <?php echo ($post['status'] ?? '') === 'published' ? 'selected' : ''; ?>>Publicado</option>
                            </select>
                        </div>
                    </div>

                    <div>
                        <label class="flex items-center">
                            <input type="checkbox" name="featured" value="1" 
                                   <?php echo ($post['featured'] ?? 0) ? 'checked' : ''; ?>
                                   class="rounded border-gray-300 text-blue-600 shadow-sm focus:border-blue-300 focus:ring focus:ring-blue-200 focus:ring-opacity-50">
                            <span class="ml-2 text-sm text-gray-700">Marcar como destaque</span>
                        </label>
                    </div>

                    <div class="flex justify-between">
                        <a href="?action=list" class="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded">
                            Cancelar
                        </a>
                        <button type="submit" name="save_post" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded">
                            <i class="fas fa-save mr-2"></i>
                            Salvar
                        </button>
                    </div>
                </form>
            </div>
        <?php endif; ?>
    </div>

<!-- TinyMCE 6 CDN com sua API Key -->
<script src="https://cdn.tiny.cloud/1/h8apq7gn7djttwhj1v86hxx3886vahl4omj06i1gtgh760zz/tinymce/6/tinymce.min.js" referrerpolicy="origin"></script>

<script>
  tinymce.init({
    selector: '#content', // O seletor do seu textarea

    // Lista de plugins (incluindo os premium do seu trial)
    plugins: [
      'anchor', 'autolink', 'charmap', 'codesample', 'emoticons', 'link', 'lists', 'media', 'searchreplace', 'table', 'visualblocks', 'wordcount',
      'checklist', 'mediaembed', 'casechange', 'formatpainter', 'pageembed', 'a11ychecker', 'tinymcespellchecker', 'permanentpen', 'powerpaste', 'advtable', 'advcode', 'advtemplate', 'importword', 'exportword', 'exportpdf'
    ],

    // Configuração da barra de ferramentas
    toolbar: 'undo redo | blocks fontfamily fontsize | bold italic underline strikethrough | link media table | spellcheckdialog a11ycheck | align lineheight | checklist numlist bullist indent outdent | emoticons charmap | removeformat',
    
    // Configurações adicionais para uma melhor experiência
    height: 650, // Aumenta a altura do editor
    menubar: 'file edit view insert format tools table help', // Adiciona o menu superior
    content_style: 'body { font-family:Helvetica,Arial,sans-serif; font-size:16px }', // Estilo do conteúdo dentro do editor
    
    // O PowerPaste é ativado automaticamente com o plugin
    powerpaste_allow_local_images: true,
    powerpaste_word_import: 'prompt',
    powerpaste_html_import: 'prompt',
  });
</script>

</body>
</html>
