<?php
/**
 * Gerenciamento de Páginas
 */

require_once __DIR__ . '/../../config/config.php';
require_once 'auth.php';

// Protege a página
requireAdminAuth();

$user = getAdminUser();
$db = Database::getInstance()->getConnection();

// Processar ações
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

// Deletar página
if ($action === 'delete' && isset($_GET['id'])) {
    try {
        $stmt = $db->prepare("DELETE FROM pages WHERE id = ?");
        $stmt->execute([$_GET['id']]);
        $message = "Página deletada com sucesso!";
        $action = 'list';
    } catch (Exception $e) {
        $error = "Erro ao deletar: " . $e->getMessage();
    }
}

// Salvar página (criar/editar)
if ($_POST && isset($_POST['save_page'])) {
    try {
        $title = $_POST['title'];
        $content = $_POST['content'];
        $parent_id = $_POST['parent_id'] ?: null;
        $status = $_POST['status'];
        $show_in_menu = isset($_POST['show_in_menu']) ? 1 : 0;
        $menu_order = $_POST['menu_order'] ?: 0;
        $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title)));
        
        if (isset($_POST['id']) && $_POST['id']) {
            // Editar
            $stmt = $db->prepare("UPDATE pages SET title=?, content=?, parent_id=?, status=?, show_in_menu=?, menu_order=?, slug=?, updated_at=NOW() WHERE id=?");
            $stmt->execute([$title, $content, $parent_id, $status, $show_in_menu, $menu_order, $slug, $_POST['id']]);
            $message = "Página atualizada com sucesso!";
        } else {
            // Criar
            $stmt = $db->prepare("INSERT INTO pages (title, content, parent_id, status, show_in_menu, menu_order, slug, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW())");
            $stmt->execute([$title, $content, $parent_id, $status, $show_in_menu, $menu_order, $slug]);
            $message = "Página criada com sucesso!";
        }
        $action = 'list';
    } catch (Exception $e) {
        $error = "Erro ao salvar: " . $e->getMessage();
    }
}

// Buscar páginas para o select de páginas pai
$parent_pages = [];
try {
    $stmt = $db->query("SELECT id, title FROM pages WHERE parent_id IS NULL ORDER BY title");
    $parent_pages = $stmt->fetchAll();
} catch (Exception $e) {
    // Ignorar erro
}

// Buscar página para edição
$page = null;
if ($action === 'edit' && isset($_GET['id'])) {
    try {
        $stmt = $db->prepare("SELECT * FROM pages WHERE id = ?");
        $stmt->execute([$_GET['id']]);
        $page = $stmt->fetch();
        if (!$page) {
            $error = "Página não encontrada!";
            $action = 'list';
        }
    } catch (Exception $e) {
        $error = "Erro ao carregar página: " . $e->getMessage();
        $action = 'list';
    }
}

// Listar páginas
$pages = [];
if ($action === 'list') {
    try {
        $stmt = $db->query("
            SELECT p.*, pp.title as parent_title 
            FROM pages p 
            LEFT JOIN pages pp ON p.parent_id = pp.id 
            ORDER BY p.menu_order, p.title
        ");
        $pages = $stmt->fetchAll();
    } catch (Exception $e) {
        $error = "Erro ao carregar páginas: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Páginas - Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script src="https://cdn.ckeditor.com/4.16.2/standard/ckeditor.js"></script>
</head>
<body class="bg-gray-100">
    <!-- Header -->
    <header class="bg-blue-900 text-white shadow-lg">
        <div class="container mx-auto px-4 py-3">
            <div class="flex justify-between items-center">
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-white hover:text-gray-300">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Voltar ao Dashboard
                    </a>
                    <h1 class="text-xl font-bold">Gerenciar Páginas</h1>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-sm">
                        <i class="fas fa-user mr-1"></i>
                        <?php echo htmlspecialchars($user['name']); ?>
                    </span>
                    <a href="?action=logout" class="bg-red-600 hover:bg-red-700 px-3 py-1 rounded text-sm">
                        <i class="fas fa-sign-out-alt mr-1"></i>
                        Sair
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="container mx-auto px-4 py-6">
        <!-- Mensagens -->
        <?php if ($message): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <?php if ($action === 'list'): ?>
            <!-- Lista de Páginas -->
            <div class="bg-white rounded-lg shadow">
                <div class="p-6 border-b flex justify-between items-center">
                    <h2 class="text-xl font-semibold">Páginas</h2>
                    <a href="?action=new" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded">
                        <i class="fas fa-plus mr-2"></i>
                        Nova Página
                    </a>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Título</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Página Pai</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Menu</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ordem</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Data</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ações</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            <?php foreach ($pages as $page): ?>
                                <tr>
                                    <td class="px-6 py-4">
                                        <div class="font-medium text-gray-900">
                                            <?php echo htmlspecialchars($page['title']); ?>
                                        </div>
                                        <div class="text-sm text-gray-500">
                                            Slug: <?php echo htmlspecialchars($page['slug']); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <?php echo htmlspecialchars($page['parent_title'] ?? '-'); ?>
                                    </td>
                                    <td class="px-6 py-4">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                            <?php echo $page['status'] === 'published' ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'; ?>">
                                            <?php echo $page['status'] === 'published' ? 'Publicado' : 'Rascunho'; ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <?php echo $page['show_in_menu'] ? 'Sim' : 'Não'; ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <?php echo $page['menu_order']; ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-500">
                                        <?php echo date('d/m/Y H:i', strtotime($page['created_at'])); ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm space-x-2">
                                        <a href="?action=edit&id=<?php echo $page['id']; ?>" 
                                           class="text-blue-600 hover:text-blue-900">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <a href="?action=delete&id=<?php echo $page['id']; ?>" 
                                           class="text-red-600 hover:text-red-900"
                                           onclick="return confirm('Tem certeza que deseja deletar esta página?')">
                                            <i class="fas fa-trash"></i>
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

        <?php else: ?>
            <!-- Formulário de Criação/Edição -->
            <div class="bg-white rounded-lg shadow">
                <div class="p-6 border-b">
                    <h2 class="text-xl font-semibold">
                        <?php echo $action === 'edit' ? 'Editar Página' : 'Nova Página'; ?>
                    </h2>
                </div>
                <form method="POST" class="p-6 space-y-6">
                    <?php if ($page): ?>
                        <input type="hidden" name="id" value="<?php echo $page['id']; ?>">
                    <?php endif; ?>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Título</label>
                        <input type="text" name="title" required
                               value="<?php echo htmlspecialchars($page['title'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Conteúdo</label>
                        <textarea name="content" id="content" rows="10"
                                  class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"><?php echo htmlspecialchars($page['content'] ?? ''); ?></textarea>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Página Pai</label>
                            <select name="parent_id" 
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="">Nenhuma (página principal)</option>
                                <?php foreach ($parent_pages as $parent): ?>
                                    <?php if (!$page || $parent['id'] != $page['id']): // Evita que página seja pai de si mesma ?>
                                        <option value="<?php echo $parent['id']; ?>"
                                                <?php echo ($page['parent_id'] ?? '') == $parent['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($parent['title']); ?>
                                        </option>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                            <select name="status" 
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="draft" <?php echo ($page['status'] ?? '') === 'draft' ? 'selected' : ''; ?>>Rascunho</option>
                                <option value="published" <?php echo ($page['status'] ?? '') === 'published' ? 'selected' : ''; ?>>Publicado</option>
                            </select>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Ordem no Menu</label>
                            <input type="number" name="menu_order" min="0"
                                   value="<?php echo htmlspecialchars($page['menu_order'] ?? '0'); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                    </div>

                    <div>
                        <label class="flex items-center">
                            <input type="checkbox" name="show_in_menu" value="1" 
                                   <?php echo ($page['show_in_menu'] ?? 1) ? 'checked' : ''; ?>
                                   class="rounded border-gray-300 text-blue-600 shadow-sm focus:border-blue-300 focus:ring focus:ring-blue-200 focus:ring-opacity-50">
                            <span class="ml-2 text-sm text-gray-700">Mostrar no menu de navegação</span>
                        </label>
                    </div>

                    <div class="flex justify-between">
                        <a href="?action=list" class="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded">
                            Cancelar
                        </a>
                        <button type="submit" name="save_page" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded">
                            <i class="fas fa-save mr-2"></i>
                            Salvar
                        </button>
                    </div>
                </form>
            </div>
        <?php endif; ?>
    </div>

<!-- TinyMCE 6 CDN com sua API Key -->
<script src="https://cdn.tiny.cloud/1/h8apq7gn7djttwhj1v86hxx3886vahl4omj06i1gtgh760zz/tinymce/6/tinymce.min.js" referrerpolicy="origin"></script>

<script>
  tinymce.init({
    selector: '#content', // O seletor do seu textarea

    // Lista de plugins (incluindo os premium do seu trial)
    plugins: [
      'anchor', 'autolink', 'charmap', 'codesample', 'emoticons', 'link', 'lists', 'media', 'searchreplace', 'table', 'visualblocks', 'wordcount',
      'checklist', 'mediaembed', 'casechange', 'formatpainter', 'pageembed', 'a11ychecker', 'tinymcespellchecker', 'permanentpen', 'powerpaste', 'advtable', 'advcode', 'advtemplate', 'importword', 'exportword', 'exportpdf'
    ],

    // Configuração da barra de ferramentas
    toolbar: 'undo redo | blocks fontfamily fontsize | bold italic underline strikethrough | link media table | spellcheckdialog a11ycheck | align lineheight | checklist numlist bullist indent outdent | emoticons charmap | removeformat',
    
    // Configurações adicionais para uma melhor experiência
    height: 650, // Aumenta a altura do editor
    menubar: 'file edit view insert format tools table help', // Adiciona o menu superior
    content_style: 'body { font-family:Helvetica,Arial,sans-serif; font-size:16px }', // Estilo do conteúdo dentro do editor
    
    // O PowerPaste é ativado automaticamente com o plugin
    powerpaste_allow_local_images: true,
    powerpaste_word_import: 'prompt',
    powerpaste_html_import: 'prompt',
  });
</script>

</body>
</html>
