<?php
/**
 * Gerenciamento de Categorias
 */

require_once __DIR__ . '/../../config/config.php';
require_once 'auth.php';

// Protege a página
requireAdminAuth();

$user = getAdminUser();
$db = Database::getInstance()->getConnection();

// Processar ações
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

// Deletar categoria
if ($action === 'delete' && isset($_GET['id'])) {
    try {
        // Verificar se há posts usando esta categoria
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM posts WHERE category_id = ?");
        $stmt->execute([$_GET['id']]);
        $count = $stmt->fetch()['count'];
        
        if ($count > 0) {
            $error = "Não é possível deletar esta categoria pois ela possui $count notícia(s) associada(s).";
        } else {
            $stmt = $db->prepare("DELETE FROM categories WHERE id = ?");
            $stmt->execute([$_GET['id']]);
            $message = "Categoria deletada com sucesso!";
        }
        $action = 'list';
    } catch (Exception $e) {
        $error = "Erro ao deletar: " . $e->getMessage();
    }
}

// Salvar categoria (criar/editar)
if ($_POST && isset($_POST['save_category'])) {
    try {
        $name = $_POST['name'];
        $description = $_POST['description'];
        $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $name)));
        
        if (isset($_POST['id']) && $_POST['id']) {
            // Editar
            $stmt = $db->prepare("UPDATE categories SET name=?, description=?, slug=? WHERE id=?");
            $stmt->execute([$name, $description, $slug, $_POST['id']]);
            $message = "Categoria atualizada com sucesso!";
        } else {
            // Criar
            $stmt = $db->prepare("INSERT INTO categories (name, description, slug) VALUES (?, ?, ?)");
            $stmt->execute([$name, $description, $slug]);
            $message = "Categoria criada com sucesso!";
        }
        $action = 'list';
    } catch (Exception $e) {
        $error = "Erro ao salvar: " . $e->getMessage();
    }
}

// Buscar categoria para edição
$category = null;
if ($action === 'edit' && isset($_GET['id'])) {
    try {
        $stmt = $db->prepare("SELECT * FROM categories WHERE id = ?");
        $stmt->execute([$_GET['id']]);
        $category = $stmt->fetch();
        if (!$category) {
            $error = "Categoria não encontrada!";
            $action = 'list';
        }
    } catch (Exception $e) {
        $error = "Erro ao carregar categoria: " . $e->getMessage();
        $action = 'list';
    }
}

// Listar categorias
$categories = [];
if ($action === 'list') {
    try {
        $stmt = $db->query("
            SELECT c.*, COUNT(p.id) as post_count 
            FROM categories c 
            LEFT JOIN posts p ON c.id = p.category_id 
            GROUP BY c.id 
            ORDER BY c.name
        ");
        $categories = $stmt->fetchAll();
    } catch (Exception $e) {
        $error = "Erro ao carregar categorias: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Categorias - Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-100">
    <!-- Header -->
    <header class="bg-blue-900 text-white shadow-lg">
        <div class="container mx-auto px-4 py-3">
            <div class="flex justify-between items-center">
                <div class="flex items-center space-x-4">
                    <a href="index.php" class="text-white hover:text-gray-300">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Voltar ao Dashboard
                    </a>
                    <h1 class="text-xl font-bold">Gerenciar Categorias</h1>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-sm">
                        <i class="fas fa-user mr-1"></i>
                        <?php echo htmlspecialchars($user['name']); ?>
                    </span>
                    <a href="?action=logout" class="bg-red-600 hover:bg-red-700 px-3 py-1 rounded text-sm">
                        <i class="fas fa-sign-out-alt mr-1"></i>
                        Sair
                    </a>
                </div>
            </div>
        </div>
    </header>

    <div class="container mx-auto px-4 py-6">
        <!-- Mensagens -->
        <?php if ($message): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <!-- Formulário de Criação/Edição -->
            <div class="lg:col-span-1">
                <div class="bg-white rounded-lg shadow">
                    <div class="p-6 border-b">
                        <h2 class="text-xl font-semibold">
                            <?php echo $action === 'edit' ? 'Editar Categoria' : 'Nova Categoria'; ?>
                        </h2>
                    </div>
                    <form method="POST" class="p-6 space-y-4">
                        <?php if ($category): ?>
                            <input type="hidden" name="id" value="<?php echo $category['id']; ?>">
                        <?php endif; ?>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Nome</label>
                            <input type="text" name="name" required
                                   value="<?php echo htmlspecialchars($category['name'] ?? ''); ?>"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Descrição</label>
                            <textarea name="description" rows="3"
                                      class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"><?php echo htmlspecialchars($category['description'] ?? ''); ?></textarea>
                        </div>

                        <div class="flex justify-between">
                            <?php if ($action === 'edit'): ?>
                                <a href="?action=list" class="bg-gray-500 hover:bg-gray-600 text-white px-4 py-2 rounded">
                                    Cancelar
                                </a>
                            <?php else: ?>
                                <div></div>
                            <?php endif; ?>
                            <button type="submit" name="save_category" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded">
                                <i class="fas fa-save mr-2"></i>
                                <?php echo $action === 'edit' ? 'Atualizar' : 'Criar'; ?>
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Lista de Categorias -->
            <div class="lg:col-span-2">
                <div class="bg-white rounded-lg shadow">
                    <div class="p-6 border-b">
                        <h2 class="text-xl font-semibold">Categorias Existentes</h2>
                    </div>
                    <div class="overflow-x-auto">
                        <table class="w-full">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Nome</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Descrição</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Notícias</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Ações</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-gray-200">
                                <?php if (empty($categories)): ?>
                                    <tr>
                                        <td colspan="4" class="px-6 py-4 text-center text-gray-500">
                                            Nenhuma categoria encontrada.
                                        </td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($categories as $cat): ?>
                                        <tr>
                                            <td class="px-6 py-4">
                                                <div>
                                                    <div class="font-medium text-gray-900">
                                                        <?php echo htmlspecialchars($cat['name']); ?>
                                                    </div>
                                                    <div class="text-sm text-gray-500">
                                                        Slug: <?php echo htmlspecialchars($cat['slug']); ?>
                                                    </div>
                                                </div>
                                            </td>
                                            <td class="px-6 py-4 text-sm text-gray-900">
                                                <?php echo htmlspecialchars($cat['description'] ?: '-'); ?>
                                            </td>
                                            <td class="px-6 py-4">
                                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                                    <?php echo $cat['post_count']; ?> notícia(s)
                                                </span>
                                            </td>
                                            <td class="px-6 py-4 text-sm space-x-2">
                                                <a href="?action=edit&id=<?php echo $cat['id']; ?>" 
                                                   class="text-blue-600 hover:text-blue-900">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                                <?php if ($cat['post_count'] == 0): ?>
                                                    <a href="?action=delete&id=<?php echo $cat['id']; ?>" 
                                                       class="text-red-600 hover:text-red-900"
                                                       onclick="return confirm('Tem certeza que deseja deletar esta categoria?')">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                <?php else: ?>
                                                    <span class="text-gray-400" title="Não é possível deletar categoria com notícias">
                                                        <i class="fas fa-trash"></i>
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
