// Admin Panel JavaScript - Enhanced Functionality

document.addEventListener('DOMContentLoaded', function() {
    
    // Initialize all components
    initConfirmDialogs();
    initFormValidation();
    initDataTables();
    initImagePreviews();
    initTooltips();
    initAutoSave();
    
    // Confirm dialogs for dangerous actions
    function initConfirmDialogs() {
        const deleteButtons = document.querySelectorAll('.btn-danger[href*="delete"], .btn-danger[onclick*="delete"]');
        
        deleteButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                const action = this.textContent.trim();
                const confirmMessage = `Tem certeza que deseja ${action.toLowerCase()}? Esta ação não pode ser desfeita.`;
                
                if (!confirm(confirmMessage)) {
                    e.preventDefault();
                    return false;
                }
            });
        });
    }
    
    // Form validation
    function initFormValidation() {
        const forms = document.querySelectorAll('form');
        
        forms.forEach(form => {
            form.addEventListener('submit', function(e) {
                let isValid = true;
                const requiredFields = form.querySelectorAll('[required]');
                
                requiredFields.forEach(field => {
                    if (!field.value.trim()) {
                        showFieldError(field, 'Este campo é obrigatório');
                        isValid = false;
                    } else {
                        clearFieldError(field);
                    }
                });
                
                // Email validation
                const emailFields = form.querySelectorAll('input[type="email"]');
                emailFields.forEach(field => {
                    if (field.value && !isValidEmail(field.value)) {
                        showFieldError(field, 'Por favor, insira um email válido');
                        isValid = false;
                    }
                });
                
                // Password confirmation
                const passwordField = form.querySelector('input[name="password"]');
                const confirmField = form.querySelector('input[name="confirm_password"]');
                
                if (passwordField && confirmField && passwordField.value !== confirmField.value) {
                    showFieldError(confirmField, 'As senhas não coincidem');
                    isValid = false;
                }
                
                if (!isValid) {
                    e.preventDefault();
                    showAlert('Por favor, corrija os erros no formulário', 'error');
                }
            });
        });
    }
    
    // Enhanced data tables
    function initDataTables() {
        const tables = document.querySelectorAll('.table');
        
        tables.forEach(table => {
            // Add search functionality
            addTableSearch(table);
            
            // Add sorting functionality
            addTableSorting(table);
            
            // Add row highlighting
            addRowHighlighting(table);
        });
    }
    
    function addTableSearch(table) {
        const searchContainer = document.createElement('div');
        searchContainer.className = 'table-search mb-3';
        searchContainer.innerHTML = `
            <input type="text" class="form-control" placeholder="Buscar na tabela..." 
                   style="max-width: 300px; display: inline-block;">
        `;
        
        table.parentNode.insertBefore(searchContainer, table);
        
        const searchInput = searchContainer.querySelector('input');
        searchInput.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = table.querySelectorAll('tbody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(searchTerm) ? '' : 'none';
            });
        });
    }
    
    function addTableSorting(table) {
        const headers = table.querySelectorAll('th');
        
        headers.forEach((header, index) => {
            if (header.textContent.trim()) {
                header.style.cursor = 'pointer';
                header.style.userSelect = 'none';
                header.innerHTML += ' <span class="sort-indicator">↕</span>';
                
                header.addEventListener('click', function() {
                    sortTable(table, index);
                });
            }
        });
    }
    
    function sortTable(table, columnIndex) {
        const tbody = table.querySelector('tbody');
        const rows = Array.from(tbody.querySelectorAll('tr'));
        
        const isAscending = table.dataset.sortDirection !== 'asc';
        table.dataset.sortDirection = isAscending ? 'asc' : 'desc';
        
        rows.sort((a, b) => {
            const aText = a.cells[columnIndex].textContent.trim();
            const bText = b.cells[columnIndex].textContent.trim();
            
            // Try to parse as numbers
            const aNum = parseFloat(aText);
            const bNum = parseFloat(bText);
            
            if (!isNaN(aNum) && !isNaN(bNum)) {
                return isAscending ? aNum - bNum : bNum - aNum;
            }
            
            // String comparison
            return isAscending ? 
                aText.localeCompare(bText) : 
                bText.localeCompare(aText);
        });
        
        // Clear and re-append sorted rows
        tbody.innerHTML = '';
        rows.forEach(row => tbody.appendChild(row));
        
        // Update sort indicators
        const headers = table.querySelectorAll('th .sort-indicator');
        headers.forEach((indicator, index) => {
            if (index === columnIndex) {
                indicator.textContent = isAscending ? '↑' : '↓';
            } else {
                indicator.textContent = '↕';
            }
        });
    }
    
    function addRowHighlighting(table) {
        const rows = table.querySelectorAll('tbody tr');
        
        rows.forEach(row => {
            row.addEventListener('mouseenter', function() {
                this.style.backgroundColor = '#f8f9fa';
            });
            
            row.addEventListener('mouseleave', function() {
                this.style.backgroundColor = '';
            });
        });
    }
    
    // Image preview functionality
    function initImagePreviews() {
        const imageInputs = document.querySelectorAll('input[type="file"][accept*="image"]');
        
        imageInputs.forEach(input => {
            input.addEventListener('change', function(e) {
                const file = e.target.files[0];
                if (file) {
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        showImagePreview(input, e.target.result);
                    };
                    reader.readAsDataURL(file);
                }
            });
        });
    }
    
    function showImagePreview(input, src) {
        let preview = input.parentNode.querySelector('.image-preview');
        
        if (!preview) {
            preview = document.createElement('div');
            preview.className = 'image-preview mt-2';
            input.parentNode.appendChild(preview);
        }
        
        preview.innerHTML = `
            <img src="${src}" alt="Preview" style="max-width: 200px; max-height: 200px; border-radius: 4px; border: 1px solid #ddd;">
            <button type="button" class="btn btn-sm btn-danger mt-1" onclick="this.parentNode.remove()">Remover</button>
        `;
    }
    
    // Tooltips for better UX
    function initTooltips() {
        const elementsWithTooltips = document.querySelectorAll('[title]');
        
        elementsWithTooltips.forEach(element => {
            element.addEventListener('mouseenter', function(e) {
                showTooltip(e.target, e.target.getAttribute('title'));
            });
            
            element.addEventListener('mouseleave', function() {
                hideTooltip();
            });
        });
    }
    
    function showTooltip(element, text) {
        const tooltip = document.createElement('div');
        tooltip.className = 'custom-tooltip';
        tooltip.textContent = text;
        tooltip.style.cssText = `
            position: absolute;
            background: #333;
            color: white;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 12px;
            z-index: 1000;
            pointer-events: none;
        `;
        
        document.body.appendChild(tooltip);
        
        const rect = element.getBoundingClientRect();
        tooltip.style.left = rect.left + 'px';
        tooltip.style.top = (rect.top - tooltip.offsetHeight - 5) + 'px';
    }
    
    function hideTooltip() {
        const tooltip = document.querySelector('.custom-tooltip');
        if (tooltip) {
            tooltip.remove();
        }
    }
    
    // Auto-save functionality for forms
    function initAutoSave() {
        const forms = document.querySelectorAll('form[data-autosave]');
        
        forms.forEach(form => {
            const inputs = form.querySelectorAll('input, textarea, select');
            
            inputs.forEach(input => {
                input.addEventListener('input', debounce(function() {
                    autoSaveForm(form);
                }, 2000));
            });
        });
    }
    
    function autoSaveForm(form) {
        const formData = new FormData(form);
        formData.append('autosave', '1');
        
        fetch(form.action, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Rascunho salvo automaticamente', 'info', 2000);
            }
        })
        .catch(error => {
            console.log('Auto-save failed:', error);
        });
    }
    
    // Utility functions
    function showFieldError(field, message) {
        clearFieldError(field);
        
        field.classList.add('is-invalid');
        
        const errorDiv = document.createElement('div');
        errorDiv.className = 'invalid-feedback';
        errorDiv.textContent = message;
        
        field.parentNode.appendChild(errorDiv);
    }
    
    function clearFieldError(field) {
        field.classList.remove('is-invalid');
        
        const errorDiv = field.parentNode.querySelector('.invalid-feedback');
        if (errorDiv) {
            errorDiv.remove();
        }
    }
    
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    function showAlert(message, type = 'info', duration = 5000) {
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type}`;
        alertDiv.textContent = message;
        alertDiv.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 1050;
            min-width: 300px;
            animation: slideIn 0.3s ease;
        `;
        
        document.body.appendChild(alertDiv);
        
        setTimeout(() => {
            alertDiv.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => alertDiv.remove(), 300);
        }, duration);
    }
    
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
    
    // Add CSS animations
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideIn {
            from { transform: translateX(100%); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
        
        @keyframes slideOut {
            from { transform: translateX(0); opacity: 1; }
            to { transform: translateX(100%); opacity: 0; }
        }
        
        .sort-indicator {
            font-size: 0.8em;
            margin-left: 5px;
            color: #6c757d;
        }
        
        .table-search input {
            transition: all 0.3s ease;
        }
        
        .table-search input:focus {
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
    `;
    document.head.appendChild(style);
});

// Global utility functions
window.AdminUtils = {
    confirmDelete: function(message) {
        return confirm(message || 'Tem certeza que deseja excluir este item?');
    },
    
    toggleStatus: function(id, type, currentStatus) {
        const newStatus = currentStatus === 'active' ? 'inactive' : 'active';
        const action = newStatus === 'active' ? 'ativar' : 'desativar';
        
        if (confirm(`Tem certeza que deseja ${action} este item?`)) {
            window.location.href = `${type}.php?action=toggle_status&id=${id}&status=${newStatus}`;
        }
    },
    
    bulkAction: function(action, selectedIds) {
        if (selectedIds.length === 0) {
            alert('Por favor, selecione pelo menos um item.');
            return false;
        }
        
        const actionText = {
            'delete': 'excluir',
            'activate': 'ativar',
            'deactivate': 'desativar'
        };
        
        const message = `Tem certeza que deseja ${actionText[action]} ${selectedIds.length} item(s) selecionado(s)?`;
        
        return confirm(message);
    }
};
