<?php
/**
 * Model de Posts
 */

class Post {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Busca todos os posts publicados com paginao dinmica.
     * A quantidade de posts por pgina  obtida das configuraes do site.
     */
    public function getAllPublished($page = 1, $perPage = null) {
        // Se a quantidade por pgina no for especificada, busca a configurao do banco.
        // Usa 8 como um valor padro seguro caso a configurao no exista.
        if ($perPage === null) {
            $perPage = (int) get_setting('posts_per_page', 8);
        }

        $offset = ($page - 1) * $perPage;
        
        $sql = "SELECT p.*, c.name as category_name, c.color as category_color, c.slug as category_slug,
                       u.full_name as author_name,
                       (SELECT COUNT(*) FROM comments WHERE post_id = p.id AND status = 'approved') as comments_count
                FROM posts p
                LEFT JOIN categories c ON p.category_id = c.id
                LEFT JOIN users u ON p.author_id = u.id
                WHERE p.status = 'published'
                ORDER BY p.published_at DESC
                LIMIT :limit OFFSET :offset";
        
        $stmt = $this->db->prepare($sql);
        $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    public function countPublished() {
        $sql = "SELECT COUNT(*) as total FROM posts WHERE status = 'published'";
        $stmt = $this->db->query($sql);
        $result = $stmt->fetch();
        return $result['total'];
    }
    
    public function getBySlug($slug) {
        $sql = "SELECT p.*, c.name as category_name, c.color as category_color, c.slug as category_slug,
                       u.full_name as author_name,
                       (SELECT COUNT(*) FROM comments WHERE post_id = p.id AND status = 'approved') as comments_count
                FROM posts p
                LEFT JOIN categories c ON p.category_id = c.id
                LEFT JOIN users u ON p.author_id = u.id
                WHERE p.slug = :slug AND p.status = 'published'";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['slug' => $slug]);
        
        return $stmt->fetch();
    }
    
    public function incrementViews($id) {
        $sql = "UPDATE posts SET views = views + 1 WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute(['id' => $id]);
    }
    
    /**
     * Busca posts de uma categoria especfica com paginao dinmica.
     * A quantidade de posts por pgina  obtida das configuraes do site.
     */
    public function getByCategory($categorySlug, $page = 1, $perPage = null) {
        // Se a quantidade por pgina no for especificada, busca a configurao do banco.
        if ($perPage === null) {
            $perPage = (int) get_setting('posts_per_page', 8);
        }

        $offset = ($page - 1) * $perPage;
        
        $sql = "SELECT p.*, c.name as category_name, c.color as category_color, c.slug as category_slug,
                       u.full_name as author_name,
                       (SELECT COUNT(*) FROM comments WHERE post_id = p.id AND status = 'approved') as comments_count
                FROM posts p
                INNER JOIN categories c ON p.category_id = c.id
                LEFT JOIN users u ON p.author_id = u.id
                WHERE c.slug = :category_slug AND p.status = 'published'
                ORDER BY p.published_at DESC
                LIMIT :limit OFFSET :offset";
        
        $stmt = $this->db->prepare($sql);
        $stmt->bindValue(':category_slug', $categorySlug);
        $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    public function getAll($page = 1, $perPage = 20) {
        $offset = ($page - 1) * $perPage;
        
        $sql = "SELECT p.*, c.name as category_name, u.full_name as author_name
                FROM posts p
                LEFT JOIN categories c ON p.category_id = c.id
                LEFT JOIN users u ON p.author_id = u.id
                ORDER BY p.created_at DESC
                LIMIT :limit OFFSET :offset";
        
        $stmt = $this.db->prepare($sql);
        $stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    public function create($data) {
        $sql = "INSERT INTO posts (title, slug, content, excerpt, image, category_id, author_id, status, published_at)
                VALUES (:title, :slug, :content, :excerpt, :image, :category_id, :author_id, :status, :published_at)";
        
        $stmt = $this->db->prepare($sql);
        return $stmt->execute($data);
    }
    
    public function update($id, $data) {
        $sql = "UPDATE posts 
                SET title = :title, slug = :slug, content = :content, excerpt = :excerpt, 
                    image = :image, category_id = :category_id, status = :status, published_at = :published_at
                WHERE id = :id";
        
        $data['id'] = $id;
        $stmt = $this->db->prepare($sql);
        return $stmt->execute($data);
    }
    
    public function delete($id) {
        $sql = "DELETE FROM posts WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute(['id' => $id]);
    }
    
    public function getById($id) {
        $sql = "SELECT * FROM posts WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['id' => $id]);
        return $stmt->fetch();
    }
}