<?php
/**
 * Model de Páginas
 */

class Page {
    // 1. Declaração ÚNICA da propriedade $db
    private $db;
    private $table_name = "pages";
    
    // 2. Construtor ÚNICO
    public function __construct() {
        // A linha "require_once 'Database.php';" deve estar em um arquivo de inicialização,
        // mas o construtor precisa da classe Database.
        $this->db = Database::getInstance()->getConnection();
    }

    // --- MÉTODOS DA PRIMEIRA PARTE ---

    // Método para buscar todas as páginas (incluindo rascunhos)
    public function getAll() {
        $sql = "SELECT * FROM " . $this->table_name . " ORDER BY title ASC";
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Método para buscar uma página por ID
    public function getById($id) {
        $sql = "SELECT * FROM " . $this->table_name . " WHERE id = :id LIMIT 0,1";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['id' => $id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Método para criar uma nova página
    public function create($title, $slug, $content, $status, $parent_id = null, $menu_order = 0) {
        $sql = "INSERT INTO " . $this->table_name . " (title, slug, content, status, parent_id, menu_order) 
                VALUES (:title, :slug, :content, :status, :parent_id, :menu_order)";
        
        $stmt = $this->db->prepare($sql);

        $title = htmlspecialchars(strip_tags($title));
        $slug = htmlspecialchars(strip_tags($slug));
        $status = htmlspecialchars(strip_tags($status));
        $parent_id = $parent_id ? htmlspecialchars(strip_tags($parent_id)) : null;
        $menu_order = htmlspecialchars(strip_tags($menu_order));

        $stmt->bindParam(':title', $title);
        $stmt->bindParam(':slug', $slug);
        $stmt->bindParam(':content', $content);
        $stmt->bindParam(':status', $status);
        $stmt->bindParam(':parent_id', $parent_id);
        $stmt->bindParam(':menu_order', $menu_order);

        return $stmt->execute();
    }

    // Método para atualizar uma página
    public function update($id, $title, $slug, $content, $status, $parent_id = null, $menu_order = 0) {
        $sql = "UPDATE " . $this->table_name . " SET title = :title, slug = :slug, content = :content, status = :status, parent_id = :parent_id, menu_order = :menu_order, updated_at = NOW() WHERE id = :id";
        
        $stmt = $this->db->prepare($sql);

        $title = htmlspecialchars(strip_tags($title));
        $slug = htmlspecialchars(strip_tags($slug));
        $status = htmlspecialchars(strip_tags($status));
        $parent_id = $parent_id ? htmlspecialchars(strip_tags($parent_id)) : null;
        $menu_order = htmlspecialchars(strip_tags($menu_order));
        $id = htmlspecialchars(strip_tags($id));

        $stmt->bindParam(':title', $title);
        $stmt->bindParam(':slug', $slug);
        $stmt->bindParam(':content', $content);
        $stmt->bindParam(':status', $status);
        $stmt->bindParam(':parent_id', $parent_id);
        $stmt->bindParam(':menu_order', $menu_order);
        $stmt->bindParam(':id', $id);

        return $stmt->execute();
    }

    // Método para deletar uma página
    public function delete($id) {
        $sql = "DELETE FROM " . $this->table_name . " WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute(['id' => $id]);
    }

    // --- MÉTODOS DA SEGUNDA PARTE ---
    
    public function getBySlug($slug) {
        $sql = "SELECT * FROM pages WHERE slug = :slug AND status = 'published'";
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['slug' => $slug]);
        return $stmt->fetch(PDO::FETCH_ASSOC); // Adicionado FETCH_ASSOC por consistência
    }
    
    public function getAllForMenu() {
    $sql = "SELECT id, title, slug, parent_id FROM pages 
            WHERE status = 'published' AND show_in_menu = 1
            ORDER BY menu_order ASC, title ASC";
    
    $stmt = $this->db->query($sql);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getChildren($parentId) {
        $sql = "SELECT * FROM pages 
                WHERE parent_id = :parent_id AND status = 'published'
                ORDER BY menu_order ASC, title ASC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute(['parent_id' => $parentId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC); // Adicionado FETCH_ASSOC por consistência
    }
}