<?php
/**
 * Model de Instagram
 * Integração com Instagram Graph API
 */

class Instagram {
    private $accessToken;
    private $cacheFile;
    private $cacheTime;
    
    public function __construct() {
        $this->accessToken = INSTAGRAM_ACCESS_TOKEN;
        $this->cacheFile = CACHE_DIR . 'instagram_feed.json';
        $this->cacheTime = INSTAGRAM_CACHE_TIME;
    }
    
    public function getPosts($limit = INSTAGRAM_POSTS_LIMIT) {
        if (!empty($this->accessToken)) {
            $cachedData = $this->getFromCache();
            if ($cachedData !== null) {
                return array_slice($cachedData, 0, $limit);
            }
            
            $posts = $this->fetchFromAPI($limit);
            if ($posts !== null) {
                $this->saveToCache($posts);
                return $posts;
            }
        }
        
        return $this->getMockPosts($limit);
    }
    
    private function fetchFromAPI($limit) {
        if (!function_exists('curl_init')) {
            $this->logError('cURL extension not available');
            return null;
        }
        
        $url = "https://graph.instagram.com/me/media?fields=id,caption,media_type,media_url,thumbnail_url,permalink,timestamp&limit={$limit}&access_token={$this->accessToken}";
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        if ($response === false || !empty($curlError)) {
            $this->logError("Instagram API cURL error: {$curlError}");
            return null;
        }
        
        if ($httpCode === 200 && $response) {
            $data = json_decode($response, true);
            if (isset($data['data'])) {
                return $this->formatPosts($data['data']);
            }
            $this->logError("Instagram API returned invalid data structure");
        } else {
            $this->logError("Instagram API returned HTTP {$httpCode}: {$response}");
        }
        
        return null;
    }
    
    private function logError($message) {
        if (DEBUG_MODE) {
            error_log("[Instagram Feed] {$message}");
        }
    }
    
    private function formatPosts($apiPosts) {
        $formatted = [];
        
        foreach ($apiPosts as $post) {
            $imageUrl = $post['media_url'] ?? '';
            if ($post['media_type'] === 'VIDEO' && isset($post['thumbnail_url'])) {
                $imageUrl = $post['thumbnail_url'];
            }
            
            $fullCaption = $post['caption'] ?? '';
            
            $formatted[] = [
                'id' => $post['id'],
                'image' => $imageUrl,
                'caption' => $this->truncateCaption($fullCaption),
                'full_caption' => $fullCaption,
                'link' => $post['permalink'] ?? '#',
                'date' => $this->formatDate($post['timestamp'] ?? ''),
                'type' => strtolower($post['media_type'] ?? 'image')
            ];
        }
        
        return $formatted;
    }
    
    private function getFromCache() {
        if (!file_exists($this->cacheFile)) {
            return null;
        }
        
        if (!is_readable($this->cacheFile)) {
            $this->logError("Cache file is not readable");
            return null;
        }
        
        $cacheAge = time() - filemtime($this->cacheFile);
        if ($cacheAge > $this->cacheTime) {
            return null;
        }
        
        $content = @file_get_contents($this->cacheFile);
        if ($content === false) {
            $this->logError("Failed to read cache file");
            return null;
        }
        
        return json_decode($content, true);
    }
    
    private function saveToCache($data) {
        try {
            if (!is_dir(CACHE_DIR)) {
                if (!@mkdir(CACHE_DIR, 0755, true)) {
                    $this->logError("Failed to create cache directory");
                    return;
                }
            }
            
            if (!is_writable(CACHE_DIR)) {
                $this->logError("Cache directory is not writable. Please run: chmod 755 " . CACHE_DIR);
                return;
            }
            
            if (@file_put_contents($this->cacheFile, json_encode($data)) === false) {
                $this->logError("Failed to write cache file");
            }
        } catch (Exception $e) {
            $this->logError("Cache save exception: " . $e->getMessage());
        }
    }
    
    private function getMockPosts($limit) {
        $mockPosts = [
            [
                'id' => 'mock_1',
                'image' => 'https://images.unsplash.com/photo-1589829545856-d10d557cf95f?w=500',
                'caption' => 'Reunião da diretoria OAB Gaspar para planejamento estratégico 2025. #OABGaspar #Advocacia',
                'full_caption' => 'Reunião da diretoria OAB Gaspar para planejamento estratégico 2025. Discutimos as principais ações e projetos que beneficiarão os advogados de nossa região neste ano. Juntos, fortalecemos a advocacia! #OABGaspar #Advocacia #PlanejamentoEstratégico',
                'link' => SITE_INSTAGRAM,
                'date' => 'Há 2 dias',
                'type' => 'image'
            ],
            [
                'id' => 'mock_2',
                'image' => 'https://images.unsplash.com/photo-1521791136064-7986c2920216?w=500',
                'caption' => 'Palestra sobre Direitos Trabalhistas para advogados da região. Evento de grande sucesso!',
                'full_caption' => 'Palestra sobre Direitos Trabalhistas para advogados da região. Evento de grande sucesso! Agradecemos a presença de todos os colegas que participaram e contribuíram com discussões enriquecedoras. A valorização da classe é nosso compromisso! #DireitoTrabalhista #OAB #Capacitação',
                'link' => SITE_INSTAGRAM,
                'date' => 'Há 3 dias',
                'type' => 'image'
            ],
            [
                'id' => 'mock_3',
                'image' => 'https://images.unsplash.com/photo-1436450412740-6b988f486c6b?w=500',
                'caption' => 'Certificação Digital OAB - Facilite sua rotina profissional! Saiba mais em nossa sede.',
                'full_caption' => 'Certificação Digital OAB - Facilite sua rotina profissional! Com a certificação digital você pode assinar documentos, peticionar eletronicamente e muito mais, tudo com segurança e agilidade. Venha até nossa sede para saber mais sobre como adquirir a sua! #CertificaçãoDigital #OAB #TecnologiaJurídica',
                'link' => SITE_INSTAGRAM,
                'date' => 'Há 5 dias',
                'type' => 'image'
            ],
            [
                'id' => 'mock_4',
                'image' => 'https://images.unsplash.com/photo-1505664194779-8beaceb93744?w=500',
                'caption' => 'Comemoração dos 50 anos da OAB Subseção Gaspar! Obrigado por fazer parte dessa história.',
                'full_caption' => 'Comemoração dos 50 anos da OAB Subseção Gaspar! 🎉 Meio século de luta, defesa e valorização da advocacia em nossa região. Obrigado a todos os advogados, servidores e parceiros que fizeram e fazem parte dessa história. Que venham mais 50 anos de conquistas! #50AnosOABGaspar #Advocacia #História',
                'link' => SITE_INSTAGRAM,
                'date' => 'Há 1 semana',
                'type' => 'image'
            ],
            [
                'id' => 'mock_5',
                'image' => 'https://images.unsplash.com/photo-1450101499163-c8848c66ca85?w=500',
                'caption' => 'Atendimento ao público: segunda a sexta, das 8h às 18h. Estamos à disposição!',
                'full_caption' => 'Atendimento ao público: segunda a sexta, das 8h às 18h. Nossa equipe está sempre à disposição para atender advogados e cidadãos. Oferecemos diversos serviços como emissão de certidões, informações sobre processos e muito mais. Visite-nos! #AtendimentoOAB #Serviços #OABGaspar',
                'link' => SITE_INSTAGRAM,
                'date' => 'Há 1 semana',
                'type' => 'image'
            ],
            [
                'id' => 'mock_6',
                'image' => 'https://images.unsplash.com/photo-1497366216548-37526070297c?w=500',
                'caption' => 'Workshop sobre Inovação na Advocacia. Inscrições abertas! #DireitoDigital',
                'full_caption' => 'Workshop sobre Inovação na Advocacia. Inscrições abertas! 💻 Venha aprender sobre as novas tecnologias que estão transformando a prática jurídica: inteligência artificial, automação de processos, gestão de escritórios e muito mais. Não perca essa oportunidade de se atualizar! Inscreva-se pelo link na bio. #DireitoDigital #Inovação #TechLaw #OABGaspar',
                'link' => SITE_INSTAGRAM,
                'date' => 'Há 2 semanas',
                'type' => 'image'
            ]
        ];
        
        return array_slice($mockPosts, 0, $limit);
    }
    
    private function truncateCaption($caption, $length = 100) {
        if (strlen($caption) <= $length) {
            return $caption;
        }
        return substr($caption, 0, $length) . '...';
    }
    
    private function formatDate($timestamp) {
        if (empty($timestamp)) {
            return 'Recente';
        }
        
        $date = strtotime($timestamp);
        $diff = time() - $date;
        
        if ($diff < 3600) {
            $minutes = floor($diff / 60);
            return "Há {$minutes} " . ($minutes == 1 ? 'minuto' : 'minutos');
        } elseif ($diff < 86400) {
            $hours = floor($diff / 3600);
            return "Há {$hours} " . ($hours == 1 ? 'hora' : 'horas');
        } elseif ($diff < 604800) {
            $days = floor($diff / 86400);
            return "Há {$days} " . ($days == 1 ? 'dia' : 'dias');
        } elseif ($diff < 2592000) {
            $weeks = floor($diff / 604800);
            return "Há {$weeks} " . ($weeks == 1 ? 'semana' : 'semanas');
        } else {
            return date('d/m/Y', $date);
        }
    }
}
