<?php
/** Don't load directly */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! function_exists( 'foxiz_get_svg' ) ) {
	/**
	 * @param string $svg_name
	 * @param string $color
	 * @param string $ui
	 *
	 * @return false|string
	 * get svg icon
	 */
	function foxiz_get_svg( $svg_name = '', $color = '', $ui = '' ) {

		$output = Foxiz_SVG_Icons::get_svg( $svg_name, $color, $ui );

		if ( empty( $ui ) ) {
			$output = wp_kses( $output,
				array(
					'svg'     => array(
						'class'       => true,
						'xmlns'       => true,
						'width'       => true,
						'height'      => true,
						'viewbox'     => true,
						'aria-hidden' => true,
						'role'        => true,
						'focusable'   => true,
					),
					'rect'    => array(
						'width'  => true,
						'height' => true,
					),
					'path'    => array(
						'fill'              => true,
						'fill-rule'         => true,
						'd'                 => true,
						'transform'         => true,
						'stroke'            => true,
						'stroke-width'      => true,
						'stroke-miterlimit' => true,
					),
					'circle'  => array(
						'fill' => true,
						'cx'   => true,
						'cy'   => true,
						'r'    => true
					),
					'polygon' => array(
						'fill'      => true,
						'fill-rule' => true,
						'points'    => true,
						'transform' => true,
						'focusable' => true,
					)
				)
			);
		}

		if ( ! $output ) {
			return false;
		}

		return $output;
	}
}

if ( ! function_exists( 'foxiz_render_svg' ) ) {
	/**
	 * @param string $svg_name
	 * @param string $color
	 * @param string $ui
	 * render svg
	 */
	function foxiz_render_svg( $svg_name = '', $color = '', $ui = '' ) {

		echo foxiz_get_svg( $svg_name, $color, $ui );
	}
}

if ( ! class_exists( 'Foxiz_SVG_Icons', false ) ) {
	/**
	 * Class Foxiz_SVG_Icons
	 */
	class Foxiz_SVG_Icons {

		public static function get_svg( $icon, $color = '', $ui = '' ) {

			switch ( $ui ) {
				case 'reaction' :
					$data = self::$reaction_icons;
					break;
				case 'tweet' :
					$data = self::$tweet_icons;
					break;
				case 'address' :
					$data = self::$address_icons;
					break;
				case 'animation' :
					$data = self::$animation;
					break;
				default:
					$data = self::$ui_icons;
			}

			$data = apply_filters( 'foxiz_add_svg', $data );

			if ( array_key_exists( $icon, $data ) ) {
				$repl   = '<svg class="svg-icon svg-' . $icon . '" aria-hidden="true" role="img" focusable="false" ';
				$output = preg_replace( '/^<svg /', $repl, trim( $data[ $icon ] ) );
				if ( ! empty( $color ) ) {
					$output = str_replace( 'currentColor', $color, $output );
				}
				$output = preg_replace( "/([\n\t]+)/", ' ', $output );
				$output = preg_replace( '/>\s*</', '><', $output );

				return $output;
			}

			return null;
		}

		/** @var array svg data */
		public static $ui_icons = array(
			'adblock'      => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
							<path fill="currentColor" d="M496.375,139.958L372.042,15.625C361.958,5.552,348.583,0,334.333,0H177.667c-14.25,0-27.625,5.552-37.708,15.625
							L15.625,139.958C5.542,150.031,0,163.427,0,177.677v156.646c0,14.25,5.542,27.646,15.625,37.719l124.333,124.333
							C150.042,506.448,163.417,512,177.667,512h156.667c14.25,0,27.625-5.552,37.708-15.625l124.333-124.333
							C506.458,361.969,512,348.573,512,334.323V177.677C512,163.427,506.458,150.031,496.375,139.958z M490.667,334.323
							c0,8.552-3.333,16.583-9.375,22.635L356.958,481.292c-5.958,5.958-14.208,9.375-22.625,9.375H177.667
							c-8.417,0-16.667-3.417-22.625-9.375L30.708,356.958c-6.042-6.052-9.375-14.083-9.375-22.635V177.677
							c0-8.552,3.333-16.583,9.375-22.635L155.042,30.708c5.958-5.958,14.208-9.375,22.625-9.375h156.667
							c8.417,0,16.667,3.417,22.625,9.375l124.333,124.333c6.042,6.052,9.375,14.083,9.375,22.635V334.323z"/>
							<path fill="currentColor" d="M400.583,210.74c-10.896-7.26-23.896-9.76-36.75-6.99c-12.792,2.75-23.667,10.385-30.771,21.802L320,248.177V117.333
							c0-17.646-14.354-32-32-32c-4.212,0-8.215,0.863-11.901,2.349C272.41,74.082,260.085,64,245.333,64
							c-11.365,0-21.299,5.997-26.978,14.949c-4.656-2.647-9.96-4.283-15.689-4.283c-17.646,0-32,14.354-32,32v23.298
							C167.314,128.77,163.758,128,160,128c-17.646,0-32,14.354-32,32v167.24C128,393.823,182.167,448,248.75,448
							c53.729,0,100.229-34.583,115.708-86.146l40.458-139.26C406.208,218.125,404.458,213.323,400.583,210.74z M344,355.813
							c-12.729,42.375-51,70.854-95.25,70.854c-54.813,0-99.417-44.604-99.417-99.427V160c0-5.885,4.792-10.667,10.667-10.667
							s10.667,4.781,10.667,10.667v64c0,5.896,4.771,10.667,10.667,10.667c5.896,0,10.667-4.771,10.667-10.667V106.667
							C192,100.781,196.792,96,202.667,96s10.667,4.781,10.667,10.667v106.667c0,5.896,4.771,10.667,10.667,10.667
							s10.667-4.771,10.667-10.667V96c0-5.885,4.792-10.667,10.667-10.667S256,90.115,256,96v128c0,5.896,4.771,10.667,10.667,10.667
							c5.896,0,10.667-4.771,10.667-10.667V117.333c0-5.885,4.792-10.667,10.667-10.667s10.667,4.781,10.667,10.667V288
							c0,4.833,3.25,9.052,7.917,10.302c4.688,1.292,9.563-0.792,11.979-4.969l32.771-56.802c3.854-6.156,9.896-10.396,17-11.927
							c4.583-0.979,9.229-0.771,13.604,0.604L344,355.813z"/>
							</svg>',
			'mode-dark'    => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
			<path fill="currentColor" d="M507.681,209.011c-1.297-6.991-7.324-12.111-14.433-12.262c-7.104-0.122-13.347,4.711-14.936,11.643
						c-15.26,66.497-73.643,112.94-141.978,112.94c-80.321,0-145.667-65.346-145.667-145.666c0-68.335,46.443-126.718,112.942-141.976
						c6.93-1.59,11.791-7.826,11.643-14.934c-0.149-7.108-5.269-13.136-12.259-14.434C287.546,1.454,271.735,0,256,0
						C187.62,0,123.333,26.629,74.98,74.981C26.628,123.333,0,187.62,0,256s26.628,132.667,74.98,181.019
						C123.333,485.371,187.62,512,256,512s132.667-26.629,181.02-74.981C485.372,388.667,512,324.38,512,256
						C512,240.278,510.546,224.469,507.681,209.011z"/>
			</svg>',
			'mode-light'   => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 232.447 232.447">
			<path fill="currentColor" d="M116.211,194.8c-4.143,0-7.5,3.357-7.5,7.5v22.643c0,4.143,3.357,7.5,7.5,7.5s7.5-3.357,7.5-7.5V202.3
				C123.711,198.157,120.354,194.8,116.211,194.8z"/>
			<path fill="currentColor" d="M116.211,37.645c4.143,0,7.5-3.357,7.5-7.5V7.505c0-4.143-3.357-7.5-7.5-7.5s-7.5,3.357-7.5,7.5v22.641
				C108.711,34.288,112.068,37.645,116.211,37.645z"/>
			<path fill="currentColor" d="M50.054,171.78l-16.016,16.008c-2.93,2.929-2.931,7.677-0.003,10.606c1.465,1.466,3.385,2.198,5.305,2.198
				c1.919,0,3.838-0.731,5.302-2.195l16.016-16.008c2.93-2.929,2.931-7.677,0.003-10.606C57.731,168.852,52.982,168.851,50.054,171.78
				z"/>
			<path fill="currentColor" d="M177.083,62.852c1.919,0,3.838-0.731,5.302-2.195L198.4,44.649c2.93-2.929,2.931-7.677,0.003-10.606
				c-2.93-2.932-7.679-2.931-10.607-0.003l-16.016,16.008c-2.93,2.929-2.931,7.677-0.003,10.607
				C173.243,62.12,175.163,62.852,177.083,62.852z"/>
			<path fill="currentColor" d="M37.645,116.224c0-4.143-3.357-7.5-7.5-7.5H7.5c-4.143,0-7.5,3.357-7.5,7.5s3.357,7.5,7.5,7.5h22.645
				C34.287,123.724,37.645,120.366,37.645,116.224z"/>
			<path fill="currentColor" d="M224.947,108.724h-22.652c-4.143,0-7.5,3.357-7.5,7.5s3.357,7.5,7.5,7.5h22.652c4.143,0,7.5-3.357,7.5-7.5
				S229.09,108.724,224.947,108.724z"/>
			<path fill="currentColor" d="M50.052,60.655c1.465,1.465,3.384,2.197,5.304,2.197c1.919,0,3.839-0.732,5.303-2.196c2.93-2.929,2.93-7.678,0.001-10.606
				L44.652,34.042c-2.93-2.93-7.679-2.929-10.606-0.001c-2.93,2.929-2.93,7.678-0.001,10.606L50.052,60.655z"/>
			<path fill="currentColor" d="M182.395,171.782c-2.93-2.929-7.679-2.93-10.606-0.001c-2.93,2.929-2.93,7.678-0.001,10.607l16.007,16.008
				c1.465,1.465,3.384,2.197,5.304,2.197c1.919,0,3.839-0.732,5.303-2.196c2.93-2.929,2.93-7.678,0.001-10.607L182.395,171.782z"/>
			<path fill="currentColor" d="M116.22,48.7c-37.232,0-67.523,30.291-67.523,67.523s30.291,67.523,67.523,67.523s67.522-30.291,67.522-67.523
				S153.452,48.7,116.22,48.7z M116.22,168.747c-28.962,0-52.523-23.561-52.523-52.523S87.258,63.7,116.22,63.7
				c28.961,0,52.522,23.562,52.522,52.523S145.181,168.747,116.22,168.747z"/>
			</svg>',
			'notification' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 454.136 454.136">
				<path fill="currentColor" d="M391.024,321.948c-27.089-27.089-42.308-63.83-42.308-102.14v-35.833c0-57-39.206-104.837-92.121-118.031V29.526
							C256.595,13.22,243.375,0,227.068,0s-29.526,13.22-29.526,29.526v36.417c-52.915,13.195-92.121,61.031-92.121,118.031v35.833
							c0,38.31-15.219,75.05-42.308,102.14c-11.907,11.907-15.469,29.814-9.025,45.372c6.444,15.557,21.625,25.701,38.464,25.701h73.4
							c0,33.754,27.363,61.116,61.116,61.116s61.116-27.363,61.116-61.116h73.4c16.839,0,32.02-10.143,38.464-25.701
							C406.493,351.762,402.931,333.855,391.024,321.948z M227.068,424.137c-17.158,0-31.116-13.959-31.116-31.116h62.232
							C258.184,410.178,244.226,424.137,227.068,424.137z M372.332,355.839c-1.807,4.363-6.026,7.181-10.748,7.181H92.552
							c-4.722,0-8.941-2.819-10.748-7.181s-0.817-9.339,2.522-12.678c32.949-32.949,51.094-76.757,51.094-123.353v-35.833
							c0-50.535,41.113-91.648,91.648-91.648c50.535,0,91.648,41.113,91.648,91.648v35.833c0,46.596,18.146,90.404,51.095,123.353
							C373.15,346.5,374.139,351.476,372.332,355.839z"/>
				</svg>',
			'user'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 456.368 456.368">
							<path fill="currentColor" d="M324.194,220.526c-6.172,7.772-13.106,14.947-21.07,21.423c45.459,26.076,76.149,75.1,76.149,131.158
											c0,30.29-66.367,54.018-151.09,54.018s-151.09-23.728-151.09-54.018c0-56.058,30.69-105.082,76.149-131.158
											c-7.963-6.476-14.897-13.65-21.07-21.423c-50.624,31.969-84.322,88.41-84.322,152.581c0,19.439,10.644,46.53,61.355,65.201
											c31.632,11.647,73.886,18.06,118.979,18.06c45.093,0,87.347-6.413,118.979-18.06c50.71-18.671,61.355-45.762,61.355-65.201
											C408.516,308.936,374.818,252.495,324.194,220.526z"/>
										<path fill="currentColor" d="M228.182,239.795c56.833,0,100.597-54.936,100.597-119.897C328.779,54.907,284.993,0,228.182,0
											c-56.833,0-100.597,54.936-100.597,119.897C127.585,184.888,171.372,239.795,228.182,239.795z M228.182,29.243
											c39.344,0,71.354,40.667,71.354,90.654s-32.01,90.654-71.354,90.654s-71.354-40.667-71.354-90.654S188.838,29.243,228.182,29.243
											z"/>
							</svg>',
			'logout'       => '<svg viewBox="0 0 511 512" xmlns="http://www.w3.org/2000/svg">
			 <path fill="currentColor" d="m361.5 392v40c0 44.113281-35.886719 80-80 80h-201c-44.113281 0-80-35.886719-80-80v-352c0-44.113281 35.886719-80 80-80h201c44.113281 0 80 35.886719 80 80v40c0 11.046875-8.953125 20-20 20s-20-8.953125-20-20v-40c0-22.054688-17.945312-40-40-40h-201c-22.054688 0-40 17.945312-40 40v352c0 22.054688 17.945312 40 40 40h201c22.054688 0 40-17.945312 40-40v-40c0-11.046875 8.953125-20 20-20s20 8.953125 20 20zm136.355469-170.355469-44.785157-44.785156c-7.8125-7.8125-20.476562-7.8125-28.285156 0-7.8125 7.808594-7.8125 20.472656 0 28.28125l31.855469 31.859375h-240.140625c-11.046875 0-20 8.953125-20 20s8.953125 20 20 20h240.140625l-31.855469 31.859375c-7.8125 7.808594-7.8125 20.472656 0 28.28125 3.90625 3.90625 9.023438 5.859375 14.140625 5.859375 5.121094 0 10.238281-1.953125 14.144531-5.859375l44.785157-44.785156c19.496093-19.496094 19.496093-51.214844 0-70.710938zm0 0"/>
			 </svg>',
			'chart'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
					<path fill="currentColor" d="M460.8,25.6c-28.22-0.051-51.139,22.785-51.19,51.006c-0.024,13.301,5.14,26.088,14.394,35.642l-93.542,187.051
								c-2.05-0.34-4.119-0.551-6.195-0.631c-6.61,0.013-13.154,1.312-19.268,3.823l-72.619-81.664
								c4.239-7.495,6.495-15.948,6.554-24.559c0-28.277-22.923-51.2-51.2-51.2s-51.2,22.923-51.2,51.2
								c0.05,12.222,4.511,24.015,12.561,33.212L60.348,384.922c-3.016-0.58-6.077-0.889-9.148-0.922C22.923,384,0,406.923,0,435.2
								s22.923,51.2,51.2,51.2s51.2-22.923,51.2-51.2c-0.05-12.222-4.511-24.015-12.561-33.212l88.747-155.443
								c9.527,1.882,19.399,0.872,28.348-2.901l72.619,81.664c-4.215,7.501-6.448,15.954-6.485,24.559
								c-0.085,28.277,22.768,51.269,51.045,51.354c28.277,0.085,51.269-22.768,51.354-51.045c0.04-13.34-5.128-26.169-14.404-35.756
								l93.542-187.051c2.05,0.34,4.119,0.551,6.195,0.631c28.277,0,51.2-22.923,51.2-51.2S489.077,25.6,460.8,25.6z M51.2,452.267
								c-9.426,0-17.067-7.641-17.067-17.067s7.641-17.067,17.067-17.067s17.067,7.641,17.067,17.067S60.626,452.267,51.2,452.267z
								 M187.733,213.333c-9.426,0-17.067-7.641-17.067-17.067s7.641-17.067,17.067-17.067s17.067,7.641,17.067,17.067
								S197.159,213.333,187.733,213.333z M324.267,366.933c-9.426,0-17.067-7.641-17.067-17.067c0-9.426,7.641-17.067,17.067-17.067
								s17.067,7.641,17.067,17.067C341.333,359.292,333.692,366.933,324.267,366.933z M460.8,93.867
								c-9.426,0-17.067-7.641-17.067-17.067s7.641-17.067,17.067-17.067c9.426,0,17.067,7.641,17.067,17.067
								S470.226,93.867,460.8,93.867z"/>
					</svg>',
			'play'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 191.255 191.255">
				<path fill="currentColor" d="M162.929,66.612c-2.814-1.754-6.514-0.896-8.267,1.917s-0.895,6.513,1.917,8.266c6.544,4.081,10.45,11.121,10.45,18.833
					s-3.906,14.752-10.45,18.833l-98.417,61.365c-6.943,4.329-15.359,4.542-22.512,0.573c-7.154-3.97-11.425-11.225-11.425-19.406
					V34.262c0-8.181,4.271-15.436,11.425-19.406c7.153-3.969,15.569-3.756,22.512,0.573l57.292,35.723
					c2.813,1.752,6.513,0.895,8.267-1.917c1.753-2.812,0.895-6.513-1.917-8.266L64.512,5.247c-10.696-6.669-23.661-7-34.685-0.883
					C18.806,10.48,12.226,21.657,12.226,34.262v122.73c0,12.605,6.58,23.782,17.602,29.898c5.25,2.913,10.939,4.364,16.616,4.364
					c6.241,0,12.467-1.754,18.068-5.247l98.417-61.365c10.082-6.287,16.101-17.133,16.101-29.015S173.011,72.899,162.929,66.612z"/>
				</svg>',
			'pause'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
					<path fill="currentColor" d="M124.5,0h-35c-44.112,0-80,35.888-80,80v352c0,44.112,35.888,80,80,80h35c44.112,0,80-35.888,80-80V80
								C204.5,35.888,168.612,0,124.5,0z M164.5,432c0,22.056-17.944,40-40,40h-35c-22.056,0-40-17.944-40-40V80
								c0-22.056,17.944-40,40-40h35c22.056,0,40,17.944,40,40V432z"/>
						    <path fill="currentColor" d="M482.5,352c11.046,0,20-8.954,20-20V80c0-44.112-35.888-80-80-80h-34c-44.112,0-80,35.888-80,80v352
								c0,44.112,35.888,80,80,80h34c44.112,0,80-35.888,80-80c0-11.046-8.954-20-20-20c-11.046,0-20,8.954-20,20
								c0,22.056-17.944,40-40,40h-34c-22.056,0-40-17.944-40-40V80c0-22.056,17.944-40,40-40h34c22.056,0,40,17.944,40,40v252
								C462.5,343.046,471.454,352,482.5,352z"/>
					
					</svg>',
			'newsletter'   => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
			 <path fill="currentColor" d="M506.955,1.314c-3.119-1.78-6.955-1.75-10.045,0.078L313.656,109.756c-4.754,2.811-6.329,8.943-3.518,13.697
					c2.81,4.753,8.942,6.328,13.697,3.518l131.482-77.749L210.411,303.335L88.603,266.069l158.965-94
					c4.754-2.812,6.329-8.944,3.518-13.698c-2.81-4.753-8.943-6.33-13.697-3.518L58.91,260.392c-3.41,2.017-5.309,5.856-4.84,9.791
					s3.216,7.221,7.004,8.38l145.469,44.504L270.72,439.88c0.067,0.121,0.136,0.223,0.207,0.314c1.071,1.786,2.676,3.245,4.678,4.087
					c1.253,0.527,2.57,0.784,3.878,0.784c2.563,0,5.086-0.986,6.991-2.849l73.794-72.12l138.806,42.466
					c0.96,0.293,1.945,0.438,2.925,0.438c2.116,0,4.206-0.672,5.948-1.961C510.496,409.153,512,406.17,512,403V10
					C512,6.409,510.074,3.093,506.955,1.314z M271.265,329.23c-1.158,1.673-1.779,3.659-1.779,5.694v61.171l-43.823-79.765
					l193.921-201.21L271.265,329.23z M289.486,411.309v-62.867l48.99,14.988L289.486,411.309z M492,389.483l-196.499-60.116
					L492,45.704V389.483z"/>
					<path fill="currentColor" d="M164.423,347.577c-3.906-3.905-10.236-3.905-14.143,0l-93.352,93.352c-3.905,3.905-3.905,10.237,0,14.143
					C58.882,457.024,61.441,458,64,458s5.118-0.976,7.071-2.929l93.352-93.352C168.328,357.815,168.328,351.483,164.423,347.577z"/>
					<path fill="currentColor" d="M40.071,471.928c-3.906-3.903-10.236-3.903-14.142,0.001l-23,23c-3.905,3.905-3.905,10.237,0,14.143
					C4.882,511.024,7.441,512,10,512s5.118-0.977,7.071-2.929l23-23C43.976,482.166,43.976,475.834,40.071,471.928z"/>
					<path fill="currentColor" d="M142.649,494.34c-1.859-1.86-4.439-2.93-7.069-2.93c-2.641,0-5.21,1.07-7.07,2.93c-1.86,1.86-2.93,4.43-2.93,7.07
					c0,2.63,1.069,5.21,2.93,7.07c1.86,1.86,4.44,2.93,7.07,2.93s5.21-1.07,7.069-2.93c1.86-1.86,2.931-4.44,2.931-7.07
					C145.58,498.77,144.51,496.2,142.649,494.34z"/>
					<path fill="currentColor" d="M217.051,419.935c-3.903-3.905-10.233-3.905-14.142,0l-49.446,49.445c-3.905,3.905-3.905,10.237,0,14.142
					c1.953,1.953,4.512,2.929,7.071,2.929s5.118-0.977,7.071-2.929l49.446-49.445C220.956,430.172,220.956,423.84,217.051,419.935z"/>
					<path fill="currentColor" d="M387.704,416.139c-3.906-3.904-10.236-3.904-14.142,0l-49.58,49.58c-3.905,3.905-3.905,10.237,0,14.143
					c1.953,1.952,4.512,2.929,7.071,2.929s5.118-0.977,7.071-2.929l49.58-49.58C391.609,426.377,391.609,420.045,387.704,416.139z"/>
					<path fill="currentColor" d="M283.5,136.31c-1.86-1.86-4.44-2.93-7.07-2.93s-5.21,1.07-7.07,2.93c-1.859,1.86-2.93,4.44-2.93,7.08
					c0,2.63,1.07,5.2,2.93,7.06c1.86,1.87,4.44,2.93,7.07,2.93s5.21-1.06,7.07-2.93c1.859-1.86,2.93-4.43,2.93-7.06
					C286.43,140.75,285.36,138.17,283.5,136.31z"/>
				</svg>',
			'feedback'     => '<svg viewBox="0 0 511 511.9996" xmlns="http://www.w3.org/2000/svg">
					<path fill="currentColor" d="m444.464844 4.394531c-5.855469-5.859375-15.355469-5.859375-21.210938 0l-115.605468 115.605469h-262.148438c-24.8125 0-45 20.1875-45 45v209.996094c0 24.816406 20.1875 45 45 45h16v77c0 6.097656 3.691406 11.589844 9.335938 13.890625 5.726562 2.335937 12.183593.894531 16.386718-3.398438l85.585938-87.492187h174.691406c24.8125 0 45-20.183594 45-45v-170.144532l115.605469-115.605468c5.855469-5.859375 5.855469-15.355469 0-21.214844zm-137.886719 243.949219-42.425781-42.425781 127.28125-127.277344 42.425781 42.425781zm-58.40625-15.980469 31.960937 31.960938-54.78125 22.820312zm114.328125 142.632813c0 8.273437-6.730469 15-15 15h-181c-4.035156 0-7.902344 1.628906-10.722656 4.511718l-64.277344 65.707032v-55.21875c0-8.28125-6.714844-15-15-15h-31c-8.269531 0-15-6.726563-15-15v-209.996094c0-8.273438 6.730469-15 15-15h232.144531l-45.3125 45.3125c-1.25 1.25-2.488281 2.972656-3.3125 5.011719l-41.519531 99.675781h-81c-8.285156 0-15 6.714844-15 15 0 8.28125 6.714844 14.988281 15 14.988281h90.992188.011718c1.929688 0 4-.394531 5.894532-1.207031l108.773437-45.304688c1.8125-.707031 3.640625-1.9375 5.015625-3.3125l45.3125-45.3125zm92.570312-275.144532-42.425781-42.425781 21.214844-21.210937 42.425781 42.425781zm0 0"/>
					</svg>',
			'like'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 478.2 478.2">
			<path fill="currentColor" d="M457.575,325.1c9.8-12.5,14.5-25.9,13.9-39.7c-0.6-15.2-7.4-27.1-13-34.4c6.5-16.2,9-41.7-12.7-61.5
					c-15.9-14.5-42.9-21-80.3-19.2c-26.3,1.2-48.3,6.1-49.2,6.3h-0.1c-5,0.9-10.3,2-15.7,3.2c-0.4-6.4,0.7-22.3,12.5-58.1
					c14-42.6,13.2-75.2-2.6-97c-16.6-22.9-43.1-24.7-50.9-24.7c-7.5,0-14.4,3.1-19.3,8.8c-11.1,12.9-9.8,36.7-8.4,47.7
					c-13.2,35.4-50.2,122.2-81.5,146.3c-0.6,0.4-1.1,0.9-1.6,1.4c-9.2,9.7-15.4,20.2-19.6,29.4c-5.9-3.2-12.6-5-19.8-5h-61
					c-23,0-41.6,18.7-41.6,41.6v162.5c0,23,18.7,41.6,41.6,41.6h61c8.9,0,17.2-2.8,24-7.6l23.5,2.8c3.6,0.5,67.6,8.6,133.3,7.3
					c11.9,0.9,23.1,1.4,33.5,1.4c17.9,0,33.5-1.4,46.5-4.2c30.6-6.5,51.5-19.5,62.1-38.6c8.1-14.6,8.1-29.1,6.8-38.3
					c19.9-18,23.4-37.9,22.7-51.9C461.275,337.1,459.475,330.2,457.575,325.1z M48.275,447.3c-8.1,0-14.6-6.6-14.6-14.6V270.1
					c0-8.1,6.6-14.6,14.6-14.6h61c8.1,0,14.6,6.6,14.6,14.6v162.5c0,8.1-6.6,14.6-14.6,14.6h-61V447.3z M431.975,313.4
					c-4.2,4.4-5,11.1-1.8,16.3c0,0.1,4.1,7.1,4.6,16.7c0.7,13.1-5.6,24.7-18.8,34.6c-4.7,3.6-6.6,9.8-4.6,15.4c0,0.1,4.3,13.3-2.7,25.8
					c-6.7,12-21.6,20.6-44.2,25.4c-18.1,3.9-42.7,4.6-72.9,2.2c-0.4,0-0.9,0-1.4,0c-64.3,1.4-129.3-7-130-7.1h-0.1l-10.1-1.2
					c0.6-2.8,0.9-5.8,0.9-8.8V270.1c0-4.3-0.7-8.5-1.9-12.4c1.8-6.7,6.8-21.6,18.6-34.3c44.9-35.6,88.8-155.7,90.7-160.9
					c0.8-2.1,1-4.4,0.6-6.7c-1.7-11.2-1.1-24.9,1.3-29c5.3,0.1,19.6,1.6,28.2,13.5c10.2,14.1,9.8,39.3-1.2,72.7
					c-16.8,50.9-18.2,77.7-4.9,89.5c6.6,5.9,15.4,6.2,21.8,3.9c6.1-1.4,11.9-2.6,17.4-3.5c0.4-0.1,0.9-0.2,1.3-0.3
					c30.7-6.7,85.7-10.8,104.8,6.6c16.2,14.8,4.7,34.4,3.4,36.5c-3.7,5.6-2.6,12.9,2.4,17.4c0.1,0.1,10.6,10,11.1,23.3
					C444.875,295.3,440.675,304.4,431.975,313.4z"/>
			</svg>',
			'dislike'      => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 478.174 478.174">
			 <path fill="currentColor" d="M457.525,153.074c1.9-5.1,3.7-12,4.2-20c0.7-14.1-2.8-33.9-22.7-51.9c1.3-9.2,1.3-23.8-6.8-38.3
				c-10.7-19.2-31.6-32.2-62.2-38.7c-20.5-4.4-47.4-5.3-80-2.8c-65.7-1.3-129.7,6.8-133.3,7.3l-23.5,2.8c-6.8-4.8-15.1-7.6-24-7.6h-61
				c-23,0-41.6,18.7-41.6,41.6v162.5c0,23,18.7,41.6,41.6,41.6h61c7.2,0,13.9-1.8,19.8-5c4.2,9.2,10.4,19.7,19.6,29.4
				c0.5,0.5,1,1,1.6,1.4c31.4,24.1,68.4,110.9,81.5,146.3c-1.3,11-2.6,34.8,8.4,47.7c4.9,5.7,11.7,8.8,19.3,8.8
				c7.7,0,34.3-1.8,50.9-24.7c15.7-21.8,16.6-54.4,2.6-97c-11.8-35.8-12.9-51.7-12.5-58.1c5.4,1.2,10.7,2.3,15.8,3.2h0.1
				c0.9,0.2,22.9,5.1,49.2,6.3c37.4,1.8,64.5-4.7,80.3-19.2c21.8-19.9,19.2-45.3,12.7-61.5c5.6-7.3,12.4-19.2,13-34.4
				C471.925,178.974,467.325,165.674,457.525,153.074z M109.225,222.674h-61c-8.1,0-14.6-6.6-14.6-14.6v-162.5
				c0-8.1,6.6-14.6,14.6-14.6h61c8.1,0,14.6,6.6,14.6,14.6v162.5C123.825,216.174,117.325,222.674,109.225,222.674z M430.925,232.374
				c0,0.1,3.5,5.6,4.7,13.1c1.5,9.3-1.1,17-8.1,23.4c-19.1,17.4-74.1,13.4-104.8,6.6c-0.4-0.1-0.8-0.2-1.3-0.3
				c-5.5-1-11.4-2.2-17.4-3.5c-6.4-2.3-15.2-2-21.8,3.9c-13.3,11.8-11.8,38.6,4.9,89.5c11,33.4,11.4,58.6,1.2,72.7
				c-8.6,11.9-22.8,13.4-28.2,13.5c-2.4-4-3.1-17.7-1.3-29c0.3-2.2,0.1-4.5-0.6-6.7c-1.9-5.1-45.8-125.3-90.7-160.9
				c-11.7-12.7-16.8-27.6-18.6-34.3c1.2-3.9,1.9-8.1,1.9-12.4v-162.4c0-3-0.3-6-0.9-8.8l10.1-1.2h0.1c0.6-0.1,65.7-8.5,130-7.1
				c0.4,0,0.9,0,1.4,0c30.3-2.4,54.8-1.7,72.9,2.2c22.4,4.8,37.2,13.2,44,25.1c7.1,12.3,3.2,25,2.9,26.2c-2.1,5.6-0.2,11.7,4.6,15.3
				c29.6,22.2,16,48.1,14.2,51.3c-3.3,5.2-2.5,11.8,1.8,16.3c8.6,9,12.8,18,12.5,26.8c-0.4,13.1-10.5,22.9-11.2,23.5
				C428.225,219.474,427.325,226.774,430.925,232.374z"/>
		</svg>',
			'fire'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512">
			<path fill="currentColor" d="M412.91,157.39c-7.892-8.589-18.469-19.248-30.652-26.211c-0.321-0.191-0.656-0.369-0.998-0.54
						c-6.594-3.3-14.534-2.788-20.656,1.312c-6.58,4.414-15.599,10.461-8.261,108.738c0.014,0.246,0.041,0.485,0.068,0.731
						c0.615,5.582,0.232,8.814-0.198,10.536c-0.068,0.287-0.137,0.574-0.198,0.861c-4.503,22.015-24.004,32.832-39.323,35.401
						c-7.503,1.264-33.071,3.498-47.222-20.396c-8.678-14.957-9.6-32.928-2.555-49.388c1.606-3.232,3.157-6.519,4.721-9.826
						c3.594-7.612,6.99-14.8,11.315-21.708c32.627-53.939,27.359-118.892-14.458-178.208c-3.539-5.029-9.156-8.199-15.285-8.637
						c-6.122-0.458-12.142,1.893-16.365,6.361c-16.235,17.178-32.046,35.21-47.338,52.647c-18.387,20.97-37.41,42.658-57.198,62.828
						c-29.839,31.568-70.864,80.717-82.486,142.971c-7.571,41.667-1.633,85.67,16.672,123.811
						c33.037,69.839,107.085,119.302,184.241,123.087c3.225,0.157,6.437,0.239,9.655,0.239c74.43-0.007,147.241-41.844,183.394-106.21
						C487.552,321.987,476.749,222.22,412.91,157.39z M404.116,385.587c-30.365,54.055-92.872,88.294-155.366,85.233
						c-62.48-3.068-122.438-43.122-149.243-99.78c-14.65-30.516-19.392-65.616-13.365-98.742c7.318-39.221,30.188-78.059,71.69-121.96
						c20.341-20.717,39.74-42.849,58.503-64.243c9.142-10.434,18.476-21.073,27.919-31.547c13.638,27.42,24.96,69.244-0.226,110.884
						c-5.439,8.691-9.518,17.321-13.461,25.664c-1.51,3.211-3.013,6.389-4.578,9.518c-0.157,0.314-0.301,0.629-0.444,0.95
						c-12.367,28.261-10.693,60.724,4.558,86.989c18.305,30.912,52.524,46.272,89.374,40.095c36.891-6.191,65.343-32.552,72.62-67.201
						c1.715-7.134,2.084-15.046,1.107-24.161c-1.121-15.128-1.701-29.962-1.934-42.89C435.246,246.019,440.747,321.324,404.116,385.587
						z"/>
			</svg>',
			'virus'        => '<svg viewBox="0 0 512 512" xmlns="http://www.w3.org/2000/svg">
				<path fill="currentColor" d="m480 224c-11.82 0-22.16 6.44-27.7 16h-29.06c-3.26-34.338-16.889-65.697-37.727-90.886l20.612-20.612c10.603 2.854 22.516.104 30.894-8.261 12.48-12.48 12.48-32.78 0-45.26s-32.78-12.48-45.26 0c-8.349 8.359-11.109 20.226-8.261 30.894l-20.612 20.612c-25.189-20.838-56.548-34.467-90.886-37.727v-29.06c9.56-5.54 16-15.88 16-27.7 0-17.65-14.35-32-32-32s-32 14.35-32 32c0 11.82 6.44 22.16 16 27.7v29.06c-34.338 3.26-65.697 16.889-90.886 37.727l-20.612-20.612c2.848-10.669.087-22.536-8.261-30.894-12.48-12.48-32.78-12.48-45.26 0s-12.48 32.78 0 45.26c8.374 8.36 20.286 11.117 30.894 8.261l20.612 20.612c-20.838 25.189-34.467 56.548-37.727 90.886h-29.06c-5.54-9.56-15.88-16-27.7-16-17.65 0-32 14.35-32 32s14.35 32 32 32c11.82 0 22.16-6.44 27.7-16h29.06c3.26 34.338 16.889 65.697 37.727 90.886l-20.612 20.612c-10.668-2.848-22.536-.087-30.894 8.261-12.48 12.48-12.48 32.78 0 45.26 12.503 12.503 32.803 12.457 45.26 0 8.349-8.359 11.109-20.226 8.261-30.894l20.612-20.612c25.189 20.838 56.548 34.467 90.886 37.727v29.06c-9.56 5.54-16 15.88-16 27.7 0 17.65 14.35 32 32 32s32-14.35 32-32c0-11.82-6.44-22.16-16-27.7v-29.06c34.338-3.26 65.697-16.889 90.886-37.727l20.612 20.612c-2.848 10.668-.087 22.536 8.261 30.894 12.457 12.457 32.757 12.503 45.26 0 12.48-12.48 12.48-32.78 0-45.26-8.359-8.349-20.226-11.109-30.894-8.261l-20.612-20.612c20.838-25.189 34.467-56.548 37.727-90.886h29.06c5.54 9.56 15.88 16 27.7 16 17.65 0 32-14.35 32-32s-14.35-32-32-32zm-277 24c-15.44 0-28-12.56-28-28s12.56-28 28-28 28 12.56 28 28-12.56 28-28 28zm69 96c-17.65 0-32-14.35-32-32s14.35-32 32-32 32 14.35 32 32-14.35 32-32 32zm56-112c-13.23 0-24-10.77-24-24s10.77-24 24-24 24 10.77 24 24-10.77 24-24 24z"/>
				</svg>'
		);

		/**
		 * @var string[]
		 */
		public static $reaction_icons = array(
			'icon-angry'     => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientTransform="matrix(1 0 0 -1 -290 -347)" gradientUnits="userSpaceOnUse" id="svg_angry" x1="306" x2="306" y1="-378.9995" y2="-347"><stop offset="0" style="stop-color:#FFE254"/><stop offset="0.1174" style="stop-color:#FFDC54"/><stop offset="0.2707" style="stop-color:#FECC55"/><stop offset="0.4439" style="stop-color:#FCB156"/><stop offset="0.6318" style="stop-color:#FA8B57"/><stop offset="0.8291" style="stop-color:#F85B59"/><stop offset="1" style="stop-color:#F52C5B"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_angry);"/><circle cx="9.5" cy="19.5" r="1.5" style="fill:#212731;"/><circle cx="22.5" cy="19.5" r="1.5" style="fill:#212731;"/><path d="M5,16c1,1,4.025,3,9,3" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M27,16c-1,1-4.025,3-9,3" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><rect height="1" style="fill:#212731;" width="8" x="12" y="25"/></svg>',
			'icon-cry'       => '<svg viewBox="0 0 34 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientUnits="userSpaceOnUse" id="svg_cry" x1="25.001" x2="8.9985" y1="2.1416" y2="29.8586"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="17" cy="16" r="16" style="fill:url(#svg_cry);"/><path d="M27.85,13.38c-0.08,0.271-0.379,0.41-0.64,0.33l-4.81-1.47c-0.271-0.08-0.4-0.36-0.4-0.62v-0.24  c0-0.26,0.13-0.54,0.4-0.62l4.84-1.47c0.27-0.08,0.55,0.06,0.63,0.33c0.08,0.26-0.07,0.54-0.341,0.62L23.45,11.5l4.079,1.26  C27.79,12.84,27.93,13.12,27.85,13.38z" style="fill:#212731;"/><path d="M6.041,9.62c0.08-0.271,0.381-0.41,0.641-0.33l4.864,1.47C11.816,10.84,12,11.12,12,11.38v0.24  c0,0.26-0.184,0.54-0.454,0.62l-4.867,1.47c-0.27,0.08-0.563-0.06-0.644-0.33c-0.08-0.26,0.063-0.54,0.334-0.62l4.076-1.26  l-4.082-1.26C6.104,10.16,5.961,9.88,6.041,9.62z" style="fill:#212731;"/><path d="M0.525,15.598c-0.746,0.814-0.691,2.079,0.123,2.826c0.814,0.746,2.08,0.69,2.826-0.123  C4.221,17.486,4.703,14,4.703,14S1.271,14.783,0.525,15.598z" style="fill:#2667C6;"/><path d="M33.475,15.648c0.746,0.814,0.691,2.079-0.123,2.826c-0.814,0.746-2.08,0.69-2.826-0.123  c-0.746-0.814-1.229-4.301-1.229-4.301S32.729,14.834,33.475,15.648z" style="fill:#2667C6;"/><ellipse cx="17" cy="22" rx="10" ry="5" style="fill:#F52C5B;"/><path d="M17,17c-5.523,0-10,2.239-10,5c0,1.183,0.826,2.268,2.199,3.123C9.998,23.337,13.185,22,17,22  s7.001,1.337,7.801,3.123C26.174,24.268,27,23.183,27,22C27,19.239,22.523,17,17,17z" style="fill:#212731;"/></svg>',
			'icon-dead'      => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientUnits="userSpaceOnUse" id="svg_dead" x1="24.001" x2="7.9985" y1="2.1416" y2="29.8586"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_dead);"/><path d="M23,23c0,1.104-0.896,2-2,2H11c-1.104,0-2-0.896-2-2l0,0c0-1.104,0.896-2,2-2h10  C22.104,21,23,21.896,23,23L23,23z" style="fill:#F52C5B;"/><path d="M21,21H11c-1.104,0-2,0.896-2,2s0.896,2,2,2h0.307c0.688-0.581,2.52-1,4.693-1s4.005,0.419,4.693,1  H21c1.104,0,2-0.896,2-2S22.104,21,21,21z" style="fill:#212731;"/><path d="M26.529,12.76l-2.537-1.093C23.993,11.651,24,11.636,24,11.62v-0.24c0-0.016-0.007-0.031-0.008-0.047  l2.537-1.093c0.271-0.08,0.421-0.36,0.341-0.62c-0.08-0.271-0.36-0.41-0.63-0.33l-3.269,1.296l-3.29-1.296  c-0.261-0.08-0.56,0.06-0.64,0.33c-0.08,0.26,0.062,0.54,0.322,0.62l2.638,1.123c0,0.006-0.002,0.011-0.002,0.017v0.24  c0,0.006,0.002,0.011,0.002,0.017l-2.633,1.123c-0.271,0.08-0.414,0.36-0.334,0.62c0.08,0.271,0.374,0.41,0.644,0.33l3.291-1.296  l3.24,1.296c0.261,0.08,0.56-0.06,0.64-0.33C26.93,13.12,26.79,12.84,26.529,12.76z" style="fill:#212731;"/><path d="M12.529,12.76l-2.537-1.093C9.993,11.651,10,11.636,10,11.62v-0.24c0-0.016-0.007-0.031-0.008-0.047  l2.537-1.093c0.271-0.08,0.421-0.36,0.341-0.62c-0.08-0.271-0.36-0.41-0.63-0.33l-3.269,1.296L5.682,9.29  c-0.261-0.08-0.56,0.06-0.64,0.33c-0.08,0.26,0.062,0.54,0.322,0.62l2.638,1.123C8.002,11.369,8,11.374,8,11.38v0.24  c0,0.006,0.002,0.011,0.002,0.017L5.369,12.76c-0.271,0.08-0.414,0.36-0.334,0.62c0.08,0.271,0.374,0.41,0.644,0.33l3.291-1.296  l3.24,1.296c0.261,0.08,0.56-0.06,0.64-0.33C12.93,13.12,12.79,12.84,12.529,12.76z" style="fill:#212731;"/></svg>',
			'icon-embarrass' => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientUnits="userSpaceOnUse" id="svg_embarrass" x1="24.001" x2="7.9985" y1="2.1416" y2="29.8586"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_embarrass);"/><path d="M24,22c1.952,1.952-2.477,5-8,5s-9.952-3.048-8-5c2-2,2.477,1,8,1S22,20,24,22z" style="fill:#F52C5B;"/><path d="M6.538,15.078c0.94-4.015,5.941-3.985,7,0" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M19.537,15.078c0.941-4.014,5.941-3.984,7,0" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M24,22c-2-2-2.477,1-8,1s-6-3-8-1c-1.212,1.212,0.038,2.846,2.481,3.911   C11.397,25.376,13.522,25,16,25s4.603,0.376,5.519,0.911C23.962,24.846,25.213,23.212,24,22z" style="fill:#212731;"/><path d="M24,22c-2-2-2.477,1-8,1s-6-3-8-1c-1.212,1.212,0.038,2.846,2.481,3.911   C11.397,25.376,13.522,25,16,25s4.603,0.376,5.519,0.911C23.962,24.846,25.213,23.212,24,22z" style="fill:#212731;"/><path d="M30,8c0,1.104-0.896,2-2,2s-2-0.896-2-2s2-4,2-4S30,6.896,30,8z" style="fill:#2667C6;"/></svg>',
			'icon-happy'     => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientTransform="matrix(1 0 0 -1 -290 -347)" gradientUnits="userSpaceOnUse" id="svg_happy" x1="314.001" x2="297.998" y1="-349.1416" y2="-376.8595"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_happy);"/><path d="M16,27c6.075,0,11-4.925,11-11H5C5,22.075,9.925,27,16,27z" style="fill:#F52C5B;"/><path d="M16,19c3.416,0,6.468,1.557,8.484,4C26.057,21.098,27,18.659,27,16H5c0,2.659,0.944,5.098,2.515,7  C9.532,20.557,12.584,19,16,19z" style="fill:#212731;"/><path d="M26.85,13.38c-0.08,0.271-0.379,0.41-0.64,0.33l-4.81-1.47c-0.271-0.08-0.4-0.36-0.4-0.62v-0.24  c0-0.26,0.13-0.54,0.4-0.62l4.84-1.47c0.27-0.08,0.55,0.06,0.63,0.33c0.08,0.26-0.07,0.54-0.341,0.62L22.45,11.5l4.079,1.26  C26.79,12.84,26.93,13.12,26.85,13.38z" style="fill:#212731;"/><path d="M5.042,9.62c0.08-0.271,0.38-0.41,0.64-0.33l4.864,1.47C10.816,10.84,11,11.12,11,11.38v0.24  c0,0.26-0.184,0.54-0.454,0.62l-4.867,1.47c-0.27,0.08-0.563-0.06-0.644-0.33c-0.08-0.26,0.063-0.54,0.333-0.62l4.076-1.26  l-4.082-1.26C5.103,10.16,4.961,9.88,5.042,9.62z" style="fill:#212731;"/></svg>',
			'icon-joy'       => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientTransform="matrix(-1 0 0 1 69.8398 0.3862)" gradientUnits="userSpaceOnUse" id="svg_joy" x1="45.8389" x2="61.8413" y1="1.7559" y2="29.4729"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_joy);"/><radialGradient cx="44.998" cy="20" gradientTransform="matrix(-1 0 0 1 69.998 -7)" gradientUnits="userSpaceOnUse" id="svg_joy_2" r="5"><stop offset="0" style="stop-color:#F52C5B;stop-opacity:0.6"/><stop offset="0.1526" style="stop-color:#F5305B;stop-opacity:0.5389"/><stop offset="0.3108" style="stop-color:#F63D5A;stop-opacity:0.4757"/><stop offset="0.4715" style="stop-color:#F75159;stop-opacity:0.4114"/><stop offset="0.6341" style="stop-color:#F96E58;stop-opacity:0.3464"/><stop offset="0.798" style="stop-color:#FB9456;stop-opacity:0.2808"/><stop offset="0.961" style="stop-color:#FEC155;stop-opacity:0.2156"/><stop offset="1" style="stop-color:#FFCD54;stop-opacity:0.2"/></radialGradient><path d="M20,13.001C20,15.758,22.242,18,25,18c2.757,0,5-2.242,5-5s-2.24-5-5-5  C22.242,8,20,10.243,20,13.001z" style="opacity:0.6;fill:url(#svg_joy_2);"/><radialGradient cx="62.9473" cy="3.689" gradientTransform="matrix(-0.9189 0 0 0.9189 64.8423 15.6108)" gradientUnits="userSpaceOnUse" id="svg_joy_3" r="5.4414"><stop offset="0" style="stop-color:#F52C5B;stop-opacity:0.6"/><stop offset="0.1526" style="stop-color:#F5305B;stop-opacity:0.5389"/><stop offset="0.3108" style="stop-color:#F63D5A;stop-opacity:0.4757"/><stop offset="0.4715" style="stop-color:#F75159;stop-opacity:0.4114"/><stop offset="0.6341" style="stop-color:#F96E58;stop-opacity:0.3464"/><stop offset="0.798" style="stop-color:#FB9456;stop-opacity:0.2808"/><stop offset="0.961" style="stop-color:#FEC155;stop-opacity:0.2156"/><stop offset="1" style="stop-color:#FFCD54;stop-opacity:0.2"/></radialGradient><circle cx="7" cy="19" r="5" style="opacity:0.6;fill:url(#svg_joy_3);"/><path d="M22,15c0,4-6,7-10,4" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M16.293,8.222c0-2.828,3.535-4.949,6.364-2.121" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M5.687,11.758c0-2.829,3.535-4.95,6.363-2.122" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/></svg>',
			'icon-love'      => '<svg viewBox="0 0 32 33.217" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientTransform="matrix(-1 0 0 1 69.8398 0.3862)" gradientUnits="userSpaceOnUse" id="svg_love" x1="45.8389" x2="61.8413" y1="2.9731" y2="30.6902"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="17.217" r="16" style="fill:url(#svg_love);"/><path d="M17.63,25.049c5.947-1.237,9.766-7.062,8.528-13.01L4.619,16.521  C5.857,22.468,11.682,26.287,17.63,25.049z" style="fill:#F52C5B;"/><path d="M16,17.217c3.345-0.695,6.649,0.207,9.121,2.188c1.152-2.183,1.579-4.763,1.037-7.366L4.619,16.521  c0.542,2.604,1.962,4.798,3.889,6.341C9.984,20.059,12.655,17.913,16,17.217z" style="fill:#212731;"/><path d="M11.096,3.674c2.043,2.603-3.6,6.933-3.6,6.933s-6.901-1.72-6.065-4.923  c0.802-3.071,4.802-1.15,4.802-1.15S9.136,1.177,11.096,3.674z" style="fill:#F52C5B;"/><path d="M23.822,1.026c2.044,2.604-3.599,6.934-3.599,6.934s-6.901-1.721-6.065-4.923  c0.802-3.071,4.802-1.15,4.802-1.15S21.863-1.47,23.822,1.026z" style="fill:#F52C5B;"/></svg>',
			'icon-sad'       => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientUnits="userSpaceOnUse" id="svg_sad" x1="24.001" x2="7.9984" y1="2.1416" y2="29.8589"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_sad);"/><circle cx="9" cy="16" r="2" style="fill:#212731;"/><circle cx="23" cy="16" r="2" style="fill:#212731;"/><path d="M21,24c-2.211-2.212-7.789-2.212-10,0" style="fill:none;stroke:#212731;stroke-width:1.2804;stroke-miterlimit:10;"/><path d="M25,27c0,1.104-0.896,2-2,2s-2-0.896-2-2s2-4,2-4S25,25.896,25,27z" style="fill:#2667C6;"/><path d="M27,14c-1-2-3-3-5-3" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M5,14c1-2,3-3,5-3" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/></svg>',
			'icon-shy'       => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientUnits="userSpaceOnUse" id="svg_shy" x1="24.001" x2="7.9985" y1="2.1416" y2="29.8586"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_shy);"/><radialGradient cx="62.9473" cy="3.689" gradientTransform="matrix(-0.9189 0 0 0.9189 64.8423 15.6108)" gradientUnits="userSpaceOnUse" id="svg_shy_2" r="5.4414"><stop offset="0" style="stop-color:#F52C5B;stop-opacity:0.6"/><stop offset="0.1526" style="stop-color:#F5305B;stop-opacity:0.5389"/><stop offset="0.3108" style="stop-color:#F63D5A;stop-opacity:0.4757"/><stop offset="0.4715" style="stop-color:#F75159;stop-opacity:0.4114"/><stop offset="0.6341" style="stop-color:#F96E58;stop-opacity:0.3464"/><stop offset="0.798" style="stop-color:#FB9456;stop-opacity:0.2808"/><stop offset="0.961" style="stop-color:#FEC155;stop-opacity:0.2156"/><stop offset="1" style="stop-color:#FFCD54;stop-opacity:0.2"/></radialGradient><circle cx="7" cy="19" r="5" style="opacity:0.6;fill:url(#svg_shy_2);"/><radialGradient cx="43.3584" cy="3.689" gradientTransform="matrix(-0.9189 0 0 0.9189 64.8423 15.6108)" gradientUnits="userSpaceOnUse" id="svg_shy_3" r="5.4414"><stop offset="0" style="stop-color:#F52C5B;stop-opacity:0.6"/><stop offset="0.1526" style="stop-color:#F5305B;stop-opacity:0.5389"/><stop offset="0.3108" style="stop-color:#F63D5A;stop-opacity:0.4757"/><stop offset="0.4715" style="stop-color:#F75159;stop-opacity:0.4114"/><stop offset="0.6341" style="stop-color:#F96E58;stop-opacity:0.3464"/><stop offset="0.798" style="stop-color:#FB9456;stop-opacity:0.2808"/><stop offset="0.961" style="stop-color:#FEC155;stop-opacity:0.2156"/><stop offset="1" style="stop-color:#FFCD54;stop-opacity:0.2"/></radialGradient><circle cx="25" cy="19" r="5" style="opacity:0.6;fill:url(#svg_shy_3);"/><path d="M11,24.001c2.211,2.212,7.789,2.212,10,0" style="fill:none;stroke:#212731;stroke-width:1.2804;stroke-miterlimit:10;"/><path d="M27,11c-1-2-3-3-5-3" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M5,11c1-2,3-3,5-3" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><ellipse cx="9" cy="15" rx="2" ry="3" style="fill:#212731;"/><ellipse cx="23" cy="15" rx="2" ry="3" style="fill:#212731;"/></svg>',
			'icon-sleepy'    => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientUnits="userSpaceOnUse" id="svg_sleepy" x1="24.001" x2="7.9985" y1="2.1416" y2="29.8586"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_sleepy);"/><path d="M26,22c0,2.761-4.477,5-10,5S6,24.761,6,22s4.477-3,10-3S26,19.239,26,22z" style="fill:#F52C5B;"/><path d="M16,19c-5.523,0-10,0.239-10,3c0,1.183,0.826,2.268,2.199,3.123C8.998,23.337,12.185,22,16,22  s7.001,1.337,7.801,3.123C25.174,24.268,26,23.183,26,22C26,19.239,21.523,19,16,19z" style="fill:#212731;"/><path d="M26,11c-0.94,4.015-5.941,3.985-7,0" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M13,11c-0.941,4.014-5.941,3.984-7,0" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M23.098,21.845c0,0.191-0.156,0.348-0.348,0.348h-1.752c-0.132,0-0.258-0.078-0.317-0.204 c-0.024-0.048-0.03-0.096-0.03-0.149c0-0.078,0.023-0.156,0.078-0.223L22,20.094h-1.002c-0.191,0-0.348-0.156-0.348-0.348    c0-0.198,0.156-0.354,0.348-0.354h1.752c0.137,0,0.258,0.078,0.317,0.204c0.018,0.048,0.03,0.096,0.03,0.144    c0,0.084-0.024,0.162-0.078,0.229l-1.271,1.523h1.002C22.941,21.491,23.098,21.646,23.098,21.845z" style="fill:#43AF20;"/><path d="M26.926,20.286c0,0.256-0.208,0.464-0.464,0.464h-2.335c-0.176,0-0.344-0.104-0.424-0.271    c-0.031-0.064-0.039-0.128-0.039-0.2c0-0.104,0.031-0.208,0.104-0.296l1.695-2.031h-1.336c-0.256,0-0.463-0.207-0.463-0.463    c0-0.265,0.207-0.473,0.463-0.473h2.335c0.185,0,0.344,0.104,0.424,0.272c0.024,0.063,0.04,0.128,0.04,0.191    c0,0.112-0.032,0.216-0.104,0.304l-1.695,2.031h1.335C26.718,19.814,26.926,20.022,26.926,20.286z" style="fill:#43AF20;"/><path d="M31.582,18.42c0,0.32-0.26,0.58-0.58,0.58h-2.918c-0.22,0-0.43-0.13-0.529-0.34    c-0.041-0.08-0.051-0.16-0.051-0.25c0-0.13,0.041-0.26,0.131-0.37l2.118-2.538h-1.669c-0.32,0-0.58-0.26-0.58-0.58    c0-0.329,0.26-0.589,0.58-0.589h2.918c0.23,0,0.43,0.13,0.53,0.34c0.03,0.079,0.05,0.159,0.05,0.239c0,0.14-0.04,0.27-0.13,0.38 l-2.118,2.539h1.668C31.322,17.831,31.582,18.091,31.582,18.42z" style="fill:#43AF20;"/></svg>',
			'icon-surprise'  => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientTransform="matrix(1 0 0 -1 -290 -347)" gradientUnits="userSpaceOnUse" id="svg_surprise" x1="314.001" x2="297.998" y1="-349.1416" y2="-376.8595"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_surprise);"/><ellipse cx="9" cy="12" rx="2" ry="3" style="fill:#212731;"/><ellipse cx="23" cy="12" rx="2" ry="3" style="fill:#212731;"/><ellipse cx="16" cy="22.5" rx="5" ry="6.5" style="fill:#212731;"/><path d="M7,4c1-2,4-2,5,0" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/><path d="M20,4c1-2,4-2,5,0" style="fill:none;stroke:#212731;stroke-miterlimit:10;"/></svg>',
			'icon-wink'      => '<svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><linearGradient gradientUnits="userSpaceOnUse" id="svg_wink" x1="24.001" x2="7.9985" y1="2.1416" y2="29.8586"><stop offset="0" style="stop-color:#FFE254"/><stop offset="1" style="stop-color:#FFB255"/></linearGradient><circle cx="16" cy="16" r="16" style="fill:url(#svg_wink);"/><path d="M16,27c6.075,0,11-4.925,11-11H5C5,22.075,9.925,27,16,27z" style="fill:#212731;"/><path d="M10.129,21.975C10.045,22.467,10,22.977,10,23.5c0,4.142,2.686,7.5,6,7.5s6-3.358,6-7.5  c0-0.523-0.045-1.033-0.129-1.525C21.779,21.431,21.203,21,20.651,21h-9.303C10.797,21,10.221,21.431,10.129,21.975z" style="fill:#F52C5B;"/><path d="M6.042,8.62c0.08-0.271,0.38-0.41,0.64-0.33l4.864,1.47C11.816,9.84,12,10.12,12,10.38v0.24  c0,0.26-0.184,0.54-0.454,0.62l-4.867,1.47c-0.27,0.08-0.563-0.06-0.644-0.33c-0.08-0.26,0.063-0.54,0.333-0.62l4.076-1.26  L6.363,9.24C6.103,9.16,5.961,8.88,6.042,8.62z" style="fill:#212731;"/><circle cx="24" cy="11" r="2" style="fill:#212731;"/></svg>'
		);

		public static $tweet_icons = array(
			'favorite' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 455 455"><path fill="currentColor" d="M326.632,10.346c-38.733,0-74.991,17.537-99.132,46.92c-24.141-29.383-60.399-46.92-99.132-46.92 C57.586,10.346,0,67.931,0,138.714c0,55.426,33.049,119.535,98.23,190.546c50.162,54.649,104.729,96.96,120.257,108.626l9.01,6.769 l9.009-6.768c15.53-11.667,70.099-53.979,120.26-108.625C421.95,258.251,455,194.141,455,138.714 C455,67.931,397.414,10.346,326.632,10.346z"/></svg>',
			'retweet'  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 -80 512.0007 512"><path fill="currentColor" d="m178 20c0-11.046875 8.953125-20 20-20h168c44.113281 0 80 35.890625 80 80v163c0 11.046875-8.953125 20-20 20s-20-8.953125-20-20v-163c0-22.054688-17.945312-40-40-40h-168c-11.046875 0-20-8.953125-20-20zm328.046875 242.761719c-7.863281-7.753907-20.527344-7.667969-28.285156.195312l-44.496094 45.109375c-1.882813 1.890625-4.371094 2.933594-7.011719 2.933594-.003906 0-.003906 0-.007812 0-2.636719 0-5.125-1.039062-7.007813-2.925781l-45.078125-45.195313c-7.800781-7.824218-20.464844-7.839844-28.285156-.039062-7.820312 7.800781-7.835938 20.464844-.035156 28.285156l45.078125 45.199219c9.441406 9.464843 21.988281 14.675781 35.328125 14.675781h.035156c13.351562-.007812 25.90625-5.238281 35.402344-14.785156l44.554687-45.167969c7.757813-7.863281 7.671875-20.527344-.191406-28.285156zm-194.046875 49.238281h-166c-22.054688 0-40-17.941406-40-40v-168c0-11.046875-8.953125-20-20-20s-20 8.953125-20 20v168c0 44.113281 35.886719 80 80 80h166c11.046875 0 20-8.953125 20-20s-8.953125-20-20-20zm-226.253906-272h.007812c2.636719 0 5.125 1.039062 7.007813 2.925781l45.078125 45.199219c3.90625 3.917969 9.035156 5.875 14.160156 5.875 5.109375 0 10.21875-1.945312 14.125-5.839844 7.820312-7.796875 7.835938-20.460937.035156-28.28125l-45.078125-45.199218c-9.449219-9.472657-21.996093-14.7343755-35.363281-14.679688-13.351562.0117188-25.90625 5.238281-35.402344 14.785156l-44.554687 45.171875c-7.757813 7.863281-7.671875 20.523438.195312 28.28125 7.859375 7.757813 20.523438 7.671875 28.28125-.191406l44.5-45.109375c1.878907-1.890625 4.367188-2.933594 7.007813-2.9375zm0 0"/></svg>',
			'reply'    => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 511.633 511.633"><path fill="currentColor" d="M463.375,183.726c-35.782-36.735-92.789-57.764-171.02-63.094V45.83c0-7.994-3.713-13.608-11.136-16.846 c-7.803-3.23-14.466-1.902-19.985,3.999L115.06,179.161c-3.618,3.621-5.424,7.902-5.424,12.85c0,4.949,1.807,9.229,5.424,12.847 l146.178,146.177c3.432,3.617,7.71,5.425,12.85,5.425c2.283,0,4.661-0.476,7.136-1.427c7.423-3.238,11.139-8.847,11.139-16.845 v-71.663c30.642,2.475,56.097,7.471,76.376,14.989c20.27,7.519,36.494,18.034,48.677,31.549 c28.362,31.405,38.451,85.171,30.266,161.311c-0.376,4.951,1.807,8.186,6.567,9.708c0.571,0.192,1.427,0.284,2.569,0.284 c3.806,0,6.468-1.618,7.994-4.853l5.709-11.42c2.662-5.331,6.516-13.945,11.56-25.841c5.041-11.901,9.616-23.794,13.709-35.692 c4.093-11.893,7.755-25.026,10.992-39.396c3.234-14.376,4.853-27.079,4.853-38.116 C511.63,265.094,495.546,216.657,463.375,183.726z"/><path fill="currentColor" d="M63.953,192.011c0-4.952,1.809-9.233,5.424-12.85L182.725,65.531V45.83c0-7.994-3.715-13.608-11.138-16.846 c-7.804-3.23-14.465-1.902-19.983,3.999L5.424,179.161C1.809,182.781,0,187.062,0,192.011c0,4.949,1.809,9.229,5.424,12.847 l146.18,146.177c3.425,3.617,7.708,5.425,12.85,5.425c2.284,0,4.663-0.476,7.137-1.427c7.423-3.238,11.138-8.847,11.138-16.845 v-19.985L69.377,204.857C65.762,201.24,63.953,196.962,63.953,192.011z"/></svg>',
		);

		public static $address_icons = array(
			'envelope'    => '<svg xmlns="http://www.w3.org/2000/svg" height="512" viewBox="0 0 479.058 479.058" width="512"><path d="m434.146 59.882h-389.234c-24.766 0-44.912 20.146-44.912 44.912v269.47c0 24.766 20.146 44.912 44.912 44.912h389.234c24.766 0 44.912-20.146 44.912-44.912v-269.47c0-24.766-20.146-44.912-44.912-44.912zm0 29.941c2.034 0 3.969.422 5.738 1.159l-200.355 173.649-200.356-173.649c1.769-.736 3.704-1.159 5.738-1.159zm0 299.411h-389.234c-8.26 0-14.971-6.71-14.971-14.971v-251.648l199.778 173.141c2.822 2.441 6.316 3.655 9.81 3.655s6.988-1.213 9.81-3.655l199.778-173.141v251.649c-.001 8.26-6.711 14.97-14.971 14.97z"/></svg>',
			'placeholder' => '<svg xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 512 512" xml:space="preserve"><path d="M256,0C156.748,0,76,80.748,76,180c0,33.534,9.289,66.26,26.869,94.652l142.885,230.257 c2.737,4.411,7.559,7.091,12.745,7.091c0.04,0,0.079,0,0.119,0c5.231-0.041,10.063-2.804,12.75-7.292L410.611,272.22 C427.221,244.428,436,212.539,436,180C436,80.748,355.252,0,256,0z M384.866,256.818L258.272,468.186l-129.905-209.34 C113.734,235.214,105.8,207.95,105.8,180c0-82.71,67.49-150.2,150.2-150.2S406.1,97.29,406.1,180 C406.1,207.121,398.689,233.688,384.866,256.818z"/><path d="M256,90c-49.626,0-90,40.374-90,90c0,49.309,39.717,90,90,90c50.903,0,90-41.233,90-90C346,130.374,305.626,90,256,90z M256,240.2c-33.257,0-60.2-27.033-60.2-60.2c0-33.084,27.116-60.2,60.2-60.2s60.1,27.116,60.1,60.2 C316.1,212.683,289.784,240.2,256,240.2z"/></svg>',
			'telephone'   => '<svg xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 482.6 482.6" xml:space="preserve"><path d="M98.339,320.8c47.6,56.9,104.9,101.7,170.3,133.4c24.9,11.8,58.2,25.8,95.3,28.2c2.3,0.1,4.5,0.2,6.8,0.2 c24.9,0,44.9-8.6,61.2-26.3c0.1-0.1,0.3-0.3,0.4-0.5c5.8-7,12.4-13.3,19.3-20c4.7-4.5,9.5-9.2,14.1-14 c21.3-22.2,21.3-50.4-0.2-71.9l-60.1-60.1c-10.2-10.6-22.4-16.2-35.2-16.2c-12.8,0-25.1,5.6-35.6,16.1l-35.8,35.8 c-3.3-1.9-6.7-3.6-9.9-5.2c-4-2-7.7-3.9-11-6c-32.6-20.7-62.2-47.7-90.5-82.4c-14.3-18.1-23.9-33.3-30.6-48.8 c9.4-8.5,18.2-17.4,26.7-26.1c3-3.1,6.1-6.2,9.2-9.3c10.8-10.8,16.6-23.3,16.6-36s-5.7-25.2-16.6-36l-29.8-29.8 c-3.5-3.5-6.8-6.9-10.2-10.4c-6.6-6.8-13.5-13.8-20.3-20.1c-10.3-10.1-22.4-15.4-35.2-15.4c-12.7,0-24.9,5.3-35.6,15.5l-37.4,37.4 c-13.6,13.6-21.3,30.1-22.9,49.2c-1.9,23.9,2.5,49.3,13.9,80C32.739,229.6,59.139,273.7,98.339,320.8z M25.739,104.2 c1.2-13.3,6.3-24.4,15.9-34l37.2-37.2c5.8-5.6,12.2-8.5,18.4-8.5c6.1,0,12.3,2.9,18,8.7c6.7,6.2,13,12.7,19.8,19.6 c3.4,3.5,6.9,7,10.4,10.6l29.8,29.8c6.2,6.2,9.4,12.5,9.4,18.7s-3.2,12.5-9.4,18.7c-3.1,3.1-6.2,6.3-9.3,9.4 c-9.3,9.4-18,18.3-27.6,26.8c-0.2,0.2-0.3,0.3-0.5,0.5c-8.3,8.3-7,16.2-5,22.2c0.1,0.3,0.2,0.5,0.3,0.8 c7.7,18.5,18.4,36.1,35.1,57.1c30,37,61.6,65.7,96.4,87.8c4.3,2.8,8.9,5,13.2,7.2c4,2,7.7,3.9,11,6c0.4,0.2,0.7,0.4,1.1,0.6 c3.3,1.7,6.5,2.5,9.7,2.5c8,0,13.2-5.1,14.9-6.8l37.4-37.4c5.8-5.8,12.1-8.9,18.3-8.9c7.6,0,13.8,4.7,17.7,8.9l60.3,60.2 c12,12,11.9,25-0.3,37.7c-4.2,4.5-8.6,8.8-13.3,13.3c-7,6.8-14.3,13.8-20.9,21.7c-11.5,12.4-25.2,18.2-42.9,18.2 c-1.7,0-3.5-0.1-5.2-0.2c-32.8-2.1-63.3-14.9-86.2-25.8c-62.2-30.1-116.8-72.8-162.1-127c-37.3-44.9-62.4-86.7-79-131.5 C28.039,146.4,24.139,124.3,25.739,104.2z"/></svg>',
			'smartphone'  => '<svg xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 512 512" xml:space="preserve"> <path d="M302.933,42.667h-51.2c-7.074,0-12.8,5.726-12.8,12.8s5.726,12.8,12.8,12.8h51.2c7.074,0,12.8-5.726,12.8-12.8 S310.007,42.667,302.933,42.667z"/> <path d="M358.4,0H153.6c-28.228,0-51.2,22.972-51.2,51.2v409.6c0,28.228,22.972,51.2,51.2,51.2h204.8 c28.228,0,51.2-22.972,51.2-51.2V51.2C409.6,22.972,386.628,0,358.4,0z M384,460.8c0,14.14-11.46,25.6-25.6,25.6H153.6 c-14.14,0-25.6-11.46-25.6-25.6V51.2c0-14.14,11.46-25.6,25.6-25.6h204.8c14.14,0,25.6,11.46,25.6,25.6V460.8z"/> <circle cx="256" cy="443.733" r="25.6"/> <circle cx="209.067" cy="55.467" r="12.8"/></svg>'
		);

		public static $animation = array(
			'loading' => '<svg version="1.1" viewBox="0 0 66 100" xmlns="http://www.w3.org/2000/svg">
					  <circle cx="6" cy="50" r="6" fill="currentColor"><animateTransform attributeName="transform" begin="0.1" dur="1s" repeatCount="indefinite" type="translate" values="0 15 ; 0 -15; 0 15"/></circle>
					  <circle cx="30" cy="50" r="6" fill="currentColor"><animateTransform attributeName="transform" begin="0.2" dur="1s" repeatCount="indefinite" type="translate" values="0 10 ; 0 -10; 0 10"/></circle>
					  <circle cx="54" cy="50" r="6" fill="currentColor"><animateTransform attributeName="transform" begin="0.3" dur="1s" repeatCount="indefinite" type="translate" values="0 5 ; 0 -5; 0 5"/></circle>
					</svg>'
		);

		static function get_weather_icon( $name = '', $id = 'weather' ) {

			switch ( $name ) {
				case 'cloudy'   :
					return '<svg class="svg-icon svg-cloudy" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><defs><clipPath id="' . $id . '"><path fill="none" d="M41.8 20.25l4.48 6.61.22 4.64 5.31 2.45 1.69 5.97h8.08L61 27l-9.31-8.5-9.89 1.75z"/></clipPath></defs><g clip-path="url(#' . $id . ')"><path fill="none" stroke="#9ca3af" stroke-linejoin="round" stroke-width="2" d="M34.23 33.45a4.05 4.05 0 004.05 4h16.51a4.34 4.34 0 00.81-8.61 3.52 3.52 0 00.06-.66 4.06 4.06 0 00-6.13-3.48 6.08 6.08 0 00-11.25 3.19 6.34 6.34 0 00.18 1.46h-.18a4.05 4.05 0 00-4.05 4.1z"/><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="-2.1 0; 2.1 0; -2.1 0"/></g><g><path fill="none" stroke="#dddddd" stroke-linejoin="round" stroke-width="3" d="M46.5 31.5h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0h28a7 7 0 000-14z"/><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="-3 0; 3 0; -3 0"/></g></svg>';
				case 'day-cloudy'     :
					return '<svg class="svg-icon svg-day-cloudy" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><defs><clipPath id="' . $id . '"><path fill="none" d="M12 35l-8-1-1-10 2-8 5-4 4.72-2.21h6L29 10l4 3v7l-6 4h-6l-3 3v4l-4 2-2 2z"/></clipPath><clipPath id="' . $id . '-2"><path fill="none" d="M41.8 20.25l4.48 6.61.22 4.64 5.31 2.45 1.69 5.97h8.08L61 27l-9.31-8.5-9.89 1.75z"/></clipPath></defs><g clip-path="url(#' . $id . ')"><g><g><path fill="#f59e0b" d="M19 20.05A3.95 3.95 0 1115.05 24 4 4 0 0119 20.05m0-2A5.95 5.95 0 1025 24a5.95 5.95 0 00-6-5.95z"/><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M19 15.67V12.5M19 35.5v-3.17M24.89 18.11l2.24-2.24M10.87 32.13l2.24-2.24M13.11 18.11l-2.24-2.24M27.13 32.13l-2.24-2.24M10.67 24H7.5M30.5 24h-3.17"/><animateTransform attributeName="transform" dur="45s" from="0 19.22 24.293" repeatCount="indefinite" to="360 19.22 24.293" type="rotate"/></g><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="3 0; -3 0; 3 0"/></g><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="-3 0; 3 0; -3 0"/></g><g clip-path="url(#' . $id . '-2)"><path fill="none" stroke="#9ca3af" stroke-linejoin="round" stroke-width="2" d="M34.23 33.45a4.05 4.05 0 004.05 4h16.51a4.34 4.34 0 00.81-8.61 3.52 3.52 0 00.06-.66 4.06 4.06 0 00-6.13-3.48 6.08 6.08 0 00-11.25 3.19 6.34 6.34 0 00.18 1.46h-.18a4.05 4.05 0 00-4.05 4.1z"/><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="-2.1 0; 2.1 0; -2.1 0"/></g><g><path fill="none" stroke="#dddddd" stroke-linejoin="round" stroke-width="3" d="M46.5 31.5h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0h28a7 7 0 000-14z"/><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="-3 0; 3 0; -3 0"/></g></svg>';
				case 'day-rain'       :
					return '<svg class="svg-icon svg-rain" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><defs><clipPath id="' . $id . '"><path fill="none" d="M12 35l-5.28-4.21-2-6 1-7 4-5 5-3h6l5 1 3 3L33 20l-6 4h-6l-3 3v4l-4 2-2 2z"/></clipPath></defs><g clip-path="url(#' . $id . ')"><g><path fill="#f59e0b" d="M19 20.05A3.95 3.95 0 1115.05 24 4 4 0 0119 20.05m0-2A5.95 5.95 0 1025 24a5.95 5.95 0 00-6-5.95z"/><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M19 15.67V12.5M19 35.5v-3.17M24.89 18.11l2.24-2.24M10.87 32.13l2.24-2.24M13.11 18.11l-2.24-2.24M27.13 32.13l-2.24-2.24M10.67 24H7.5M30.5 24h-3.17"/><animateTransform attributeName="transform" dur="45s" from="0 19.22 24.293" repeatCount="indefinite" to="360 19.22 24.293" type="rotate"/></g></g><path fill="none" stroke="#dddddd" stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M43.67 45.5h2.83a7 7 0 000-14h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0"/><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M24.08 45.01l-.16.98"/><animateTransform attributeName="transform" dur="1.5s" repeatCount="indefinite" type="translate" values="1 -5; -2 10"/><animate attributeName="opacity" dur="1.5s" repeatCount="indefinite" values="0;1;1;0"/></g><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M31.08 45.01l-.16.98"/><animateTransform attributeName="transform" begin="-0.5s" dur="1.5s" repeatCount="indefinite" type="translate" values="1 -5; -2 10"/><animate attributeName="opacity" begin="-0.5s" dur="1.5s" repeatCount="indefinite" values="0;1;1;0"/></g><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M38.08 45.01l-.16.98"/><animateTransform attributeName="transform" begin="-1s" dur="1.5s" repeatCount="indefinite" type="translate" values="1 -5; -2 10"/><animate attributeName="opacity" begin="-1s" dur="1.5s" repeatCount="indefinite" values="0;1;1;0"/></g></svg>';
				case 'day-snow'       :
					return '<svg class="svg-icon svg-snow" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><defs><clipPath id="' . $id . '"><path fill="none" d="M12 35l-5.28-4.21-2-6 1-7 4-5 5-3h6l5 1 3 3L33 20l-6 4h-6l-3 3v4l-4 2-2 2z"/></clipPath></defs><g clip-path="url(#' . $id . ')"><g><path fill="#f59e0b" d="M19 20.05A3.95 3.95 0 1115.05 24 4 4 0 0119 20.05m0-2A5.95 5.95 0 1025 24a5.95 5.95 0 00-6-5.95z"/><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M19 15.67V12.5M19 35.5v-3.17M24.89 18.11l2.24-2.24M10.87 32.13l2.24-2.24M13.11 18.11l-2.24-2.24M27.13 32.13l-2.24-2.24M10.67 24H7.5M30.5 24h-3.17"/><animateTransform attributeName="transform" dur="45s" from="0 19.22 24.293" repeatCount="indefinite" to="360 19.22 24.293" type="rotate"/></g></g><path fill="none" stroke="#dddddd" stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M43.67 45.5h2.83a7 7 0 000-14h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0"/><g><g><g><path fill="#72b8d4" d="M24.24 42.67l.24.68a.25.25 0 00.35.14l.65-.31a.26.26 0 01.34.34l-.31.65a.25.25 0 00.14.35l.68.24a.25.25 0 010 .48l-.68.24a.25.25 0 00-.14.35l.31.65a.26.26 0 01-.34.34l-.65-.31a.25.25 0 00-.35.14l-.24.68a.25.25 0 01-.48 0l-.24-.68a.25.25 0 00-.35-.14l-.65.31a.26.26 0 01-.34-.34l.31-.65a.25.25 0 00-.14-.35l-.68-.24a.25.25 0 010-.48l.68-.24a.25.25 0 00.14-.35l-.31-.65a.26.26 0 01.34-.34l.65.31a.25.25 0 00.35-.14l.24-.68a.25.25 0 01.48 0z"/><animateTransform attributeName="transform" dur="9s" repeatCount="indefinite" type="rotate" values="0 24 45; 360 24 45"/></g><animateTransform attributeName="transform" dur="4s" repeatCount="indefinite" type="translate" values="-3 0; 3 0"/></g><animateTransform attributeName="transform" dur="4s" repeatCount="indefinite" type="translate" values="2 -6; -2 12"/><animate attributeName="opacity" dur="4s" repeatCount="indefinite" values="0;1;1;1;0"/></g><g><g><g><path fill="#72b8d4" d="M31.24 42.67l.24.68a.25.25 0 00.35.14l.65-.31a.26.26 0 01.34.34l-.31.65a.25.25 0 00.14.35l.68.24a.25.25 0 010 .48l-.68.24a.25.25 0 00-.14.35l.31.65a.26.26 0 01-.34.34l-.65-.31a.25.25 0 00-.35.14l-.24.68a.25.25 0 01-.48 0l-.24-.68a.25.25 0 00-.35-.14l-.65.31a.26.26 0 01-.34-.34l.31-.65a.25.25 0 00-.14-.35l-.68-.24a.25.25 0 010-.48l.68-.24a.25.25 0 00.14-.35l-.31-.65a.26.26 0 01.34-.34l.65.31a.25.25 0 00.35-.14l.24-.68a.25.25 0 01.48 0z"/><animateTransform attributeName="transform" dur="9s" repeatCount="indefinite" type="rotate" values="0 31 45; 360 31 45"/></g><animateTransform attributeName="transform" begin="-2s" dur="4s" repeatCount="indefinite" type="translate" values="-3 0; 3 0"/></g><animateTransform attributeName="transform" begin="-2s" dur="4s" repeatCount="indefinite" type="translate" values="2 -6; -2 12"/><animate attributeName="opacity" begin="-2s" dur="4s" repeatCount="indefinite" values="0;1;1;1;0"/></g><g><g><g><path fill="#72b8d4" d="M38.24 42.67l.24.68a.25.25 0 00.35.14l.65-.31a.26.26 0 01.34.34l-.31.65a.25.25 0 00.14.35l.68.24a.25.25 0 010 .48l-.68.24a.25.25 0 00-.14.35l.31.65a.26.26 0 01-.34.34l-.65-.31a.25.25 0 00-.35.14l-.24.68a.25.25 0 01-.48 0l-.24-.68a.25.25 0 00-.35-.14l-.65.31a.26.26 0 01-.34-.34l.31-.65a.25.25 0 00-.14-.35l-.68-.24a.25.25 0 010-.48l.68-.24a.25.25 0 00.14-.35l-.31-.65a.26.26 0 01.34-.34l.65.31a.25.25 0 00.35-.14l.24-.68a.25.25 0 01.48 0z"/><animateTransform attributeName="transform" dur="9s" repeatCount="indefinite" type="rotate" values="0 38 45; 360 38 45"/></g><animateTransform attributeName="transform" begin="-1s" dur="4s" repeatCount="indefinite" type="translate" values="-3 0; 3 0"/></g><animateTransform attributeName="transform" begin="-1s" dur="4s" repeatCount="indefinite" type="translate" values="2 -6; -2 12"/><animate attributeName="opacity" begin="-1s" dur="4s" repeatCount="indefinite" values="0;1;1;1;0"/></g></svg>';
				case 'night-fog'      :
					return '<svg class="svg-icon svg-night-fog" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><defs><clipPath id="' . $id . '"><path fill="none" d="M12 35l-5.28-4.21-2-6 1-7 4-5 5-3h6l5 1 3 3L33 20l-6 4h-6l-3 3v4l-4 2-2 2z"/></clipPath></defs><g clip-path="url(#' . $id . ')"><g><path fill="none" stroke="#72b9d5" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M29.33 26.68a10.61 10.61 0 01-10.68-10.54A10.5 10.5 0 0119 13.5a10.54 10.54 0 1011.5 13.11 11.48 11.48 0 01-1.17.07z"/><animateTransform attributeName="transform" dur="10s" repeatCount="indefinite" type="rotate" values="-10 19.22 24.293;10 19.22 24.293;-10 19.22 24.293"/></g></g><path fill="none" stroke="#dddddd" stroke-linejoin="round" stroke-width="3" d="M46.5 31.5h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0h28a7 7 0 000-14z"/><g><path fill="none" stroke="#d1d5db" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M17 58h30"/><animateTransform attributeName="transform" begin="0s" dur="5s" repeatCount="indefinite" type="translate" values="-4 0; 4 0; -4 0"/></g><g><path fill="none" stroke="#d1d5db" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M17 52h30"/><animateTransform attributeName="transform" begin="-4s" dur="5s" repeatCount="indefinite" type="translate" values="-4 0; 4 0; -4 0"/></g></svg>';
				case 'night-alt-snow' :
					return '<svg class="svg-icon svg-night-alt-snow" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><defs><clipPath id="' . $id . '"><path fill="none" d="M12 35l-5.28-4.21-2-6 1-7 4-5 5-3h6l5 1 3 3L33 20l-6 4h-6l-3 3v4l-4 2-2 2z"/></clipPath></defs><g clip-path="url(#' . $id . ')"><g><path fill="none" stroke="#72b9d5" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M29.33 26.68a10.61 10.61 0 01-10.68-10.54A10.5 10.5 0 0119 13.5a10.54 10.54 0 1011.5 13.11 11.48 11.48 0 01-1.17.07z"/><animateTransform attributeName="transform" dur="10s" repeatCount="indefinite" type="rotate" values="-10 19.22 24.293;10 19.22 24.293;-10 19.22 24.293"/></g></g><path fill="none" stroke="#dddddd" stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M43.67 45.5h2.83a7 7 0 000-14h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0"/><g><g><g><path fill="#72b8d4" d="M24.24 42.67l.24.68a.25.25 0 00.35.14l.65-.31a.26.26 0 01.34.34l-.31.65a.25.25 0 00.14.35l.68.24a.25.25 0 010 .48l-.68.24a.25.25 0 00-.14.35l.31.65a.26.26 0 01-.34.34l-.65-.31a.25.25 0 00-.35.14l-.24.68a.25.25 0 01-.48 0l-.24-.68a.25.25 0 00-.35-.14l-.65.31a.26.26 0 01-.34-.34l.31-.65a.25.25 0 00-.14-.35l-.68-.24a.25.25 0 010-.48l.68-.24a.25.25 0 00.14-.35l-.31-.65a.26.26 0 01.34-.34l.65.31a.25.25 0 00.35-.14l.24-.68a.25.25 0 01.48 0z"/><animateTransform attributeName="transform" dur="9s" repeatCount="indefinite" type="rotate" values="0 24 45; 360 24 45"/></g><animateTransform attributeName="transform" dur="4s" repeatCount="indefinite" type="translate" values="-3 0; 3 0"/></g><animateTransform attributeName="transform" dur="4s" repeatCount="indefinite" type="translate" values="2 -6; -2 12"/><animate attributeName="opacity" dur="4s" repeatCount="indefinite" values="0;1;1;1;0"/></g><g><g><g><path fill="#72b8d4" d="M31.24 42.67l.24.68a.25.25 0 00.35.14l.65-.31a.26.26 0 01.34.34l-.31.65a.25.25 0 00.14.35l.68.24a.25.25 0 010 .48l-.68.24a.25.25 0 00-.14.35l.31.65a.26.26 0 01-.34.34l-.65-.31a.25.25 0 00-.35.14l-.24.68a.25.25 0 01-.48 0l-.24-.68a.25.25 0 00-.35-.14l-.65.31a.26.26 0 01-.34-.34l.31-.65a.25.25 0 00-.14-.35l-.68-.24a.25.25 0 010-.48l.68-.24a.25.25 0 00.14-.35l-.31-.65a.26.26 0 01.34-.34l.65.31a.25.25 0 00.35-.14l.24-.68a.25.25 0 01.48 0z"/><animateTransform attributeName="transform" dur="9s" repeatCount="indefinite" type="rotate" values="0 31 45; 360 31 45"/></g><animateTransform attributeName="transform" begin="-2s" dur="4s" repeatCount="indefinite" type="translate" values="-3 0; 3 0"/></g><animateTransform attributeName="transform" begin="-2s" dur="4s" repeatCount="indefinite" type="translate" values="2 -6; -2 12"/><animate attributeName="opacity" begin="-2s" dur="4s" repeatCount="indefinite" values="0;1;1;1;0"/></g><g><g><g><path fill="#72b8d4" d="M38.24 42.67l.24.68a.25.25 0 00.35.14l.65-.31a.26.26 0 01.34.34l-.31.65a.25.25 0 00.14.35l.68.24a.25.25 0 010 .48l-.68.24a.25.25 0 00-.14.35l.31.65a.26.26 0 01-.34.34l-.65-.31a.25.25 0 00-.35.14l-.24.68a.25.25 0 01-.48 0l-.24-.68a.25.25 0 00-.35-.14l-.65.31a.26.26 0 01-.34-.34l.31-.65a.25.25 0 00-.14-.35l-.68-.24a.25.25 0 010-.48l.68-.24a.25.25 0 00.14-.35l-.31-.65a.26.26 0 01.34-.34l.65.31a.25.25 0 00.35-.14l.24-.68a.25.25 0 01.48 0z"/><animateTransform attributeName="transform" dur="9s" repeatCount="indefinite" type="rotate" values="0 38 45; 360 38 45"/></g><animateTransform attributeName="transform" begin="-1s" dur="4s" repeatCount="indefinite" type="translate" values="-3 0; 3 0"/></g><animateTransform attributeName="transform" begin="-1s" dur="4s" repeatCount="indefinite" type="translate" values="2 -6; -2 12"/><animate attributeName="opacity" begin="-1s" dur="4s" repeatCount="indefinite" values="0;1;1;1;0"/></g></svg>';
				case 'night-cloudy'   :
					return '<svg class="svg-icon svg-night-cloudy" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><defs><clipPath id="' . $id . '"><path fill="none" d="M12 35l-5.28-4.21-2-6 1-7 4-5 5-3h6l5 1 3 3L33 20l-6 4h-6l-3 3v4l-4 2-2 2z"/></clipPath><clipPath id="' . $id . '-2"><path fill="none" d="M41.8 20.25l4.48 6.61.22 4.64 5.31 2.45 1.69 5.97h8.08L61 27l-9.31-8.5-9.89 1.75z"/></clipPath></defs><g clip-path="url(#' . $id . ')"><g><g><path fill="none" stroke="#72b9d5" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M29.33 26.68a10.61 10.61 0 01-10.68-10.54A10.5 10.5 0 0119 13.5a10.54 10.54 0 1011.5 13.11 11.48 11.48 0 01-1.17.07z"/><animateTransform attributeName="transform" dur="10s" repeatCount="indefinite" type="rotate" values="-10 19.22 24.293;10 19.22 24.293;-10 19.22 24.293"/></g><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="3 0; -3 0; 3 0"/></g><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="-3 0; 3 0; -3 0"/></g><g clip-path="url(#' . $id . '-2)"><path fill="none" stroke="#9ca3af" stroke-linejoin="round" stroke-width="2" d="M34.23 33.45a4.05 4.05 0 004.05 4h16.51a4.34 4.34 0 00.81-8.61 3.52 3.52 0 00.06-.66 4.06 4.06 0 00-6.13-3.48 6.08 6.08 0 00-11.25 3.19 6.34 6.34 0 00.18 1.46h-.18a4.05 4.05 0 00-4.05 4.1z"/><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="-2.1 0; 2.1 0; -2.1 0"/></g><g><path fill="none" stroke="#dddddd" stroke-linejoin="round" stroke-width="3" d="M46.5 31.5h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0h28a7 7 0 000-14z"/><animateTransform attributeName="transform" dur="7s" repeatCount="indefinite" type="translate" values="-3 0; 3 0; -3 0"/></g></svg>';
				case 'night-rain'     :
					return '<svg class="svg-icon svg-night-rain" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><defs><clipPath id="' . $id . '"><path fill="none" d="M12 35l-5.28-4.21-2-6 1-7 4-5 5-3h6l5 1 3 3L33 20l-6 4h-6l-3 3v4l-4 2-2 2z"/></clipPath></defs><g clip-path="url(#' . $id . ')"><g><path fill="none" stroke="#72b9d5" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M29.33 26.68a10.61 10.61 0 01-10.68-10.54A10.5 10.5 0 0119 13.5a10.54 10.54 0 1011.5 13.11 11.48 11.48 0 01-1.17.07z"/><animateTransform attributeName="transform" dur="10s" repeatCount="indefinite" type="rotate" values="-10 19.22 24.293;10 19.22 24.293;-10 19.22 24.293"/></g></g><path fill="none" stroke="#dddddd" stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M43.67 45.5h2.83a7 7 0 000-14h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0"/><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M24.08 45.01l-.16.98"/><animateTransform attributeName="transform" dur="1.5s" repeatCount="indefinite" type="translate" values="1 -5; -2 10"/><animate attributeName="opacity" dur="1.5s" repeatCount="indefinite" values="0;1;1;0"/></g><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M31.08 45.01l-.16.98"/><animateTransform attributeName="transform" begin="-0.5s" dur="1.5s" repeatCount="indefinite" type="translate" values="1 -5; -2 10"/><animate attributeName="opacity" begin="-0.5s" dur="1.5s" repeatCount="indefinite" values="0;1;1;0"/></g><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M38.08 45.01l-.16.98"/><animateTransform attributeName="transform" begin="-1s" dur="1.5s" repeatCount="indefinite" type="translate" values="1 -5; -2 10"/><animate attributeName="opacity" begin="-1s" dur="1.5s" repeatCount="indefinite" values="0;1;1;0"/></g></svg>';
				case 'moon-full'      :
					return '<svg class="svg-icon svg-moon-full" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><g><path fill="none" stroke="#72b9d5" stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M46.66 36.2a16.66 16.66 0 01-16.78-16.55 16.29 16.29 0 01.55-4.15A16.56 16.56 0 1048.5 36.1c-.61.06-1.22.1-1.84.1z"/><animateTransform attributeName="transform" dur="10s" repeatCount="indefinite" type="rotate" values="-5 32 32;15 32 32;-5 32 32"/></g></svg>';
				case 'day-sunny'      :
					return '<svg class="svg-icon svg-day-sunny" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><g><path fill="#f59e0b" d="M32 23.36A8.64 8.64 0 1123.36 32 8.66 8.66 0 0132 23.36m0-3A11.64 11.64 0 1043.64 32 11.64 11.64 0 0032 20.36z"/><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M32 15.71V9.5"><animate attributeName="stroke-dasharray" calcMode="spline" dur="5s" keySplines="0.5 0 0.5 1; 0.5 0 0.5 1" keyTimes="0; .5; 1" repeatCount="indefinite" values="3 6; 6 6; 3 6"/></path><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M32 48.29v6.21"><animate attributeName="stroke-dasharray" calcMode="spline" dur="5s" keySplines="0.5 0 0.5 1; 0.5 0 0.5 1" keyTimes="0; .5; 1" repeatCount="indefinite" values="3 6; 6 6; 3 6"/></path><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M43.52 20.48l4.39-4.39"><animate attributeName="stroke-dasharray" calcMode="spline" dur="5s" keySplines="0.5 0 0.5 1; 0.5 0 0.5 1" keyTimes="0; .5; 1" repeatCount="indefinite" values="3 6; 6 6; 3 6"/></path><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M20.48 43.52l-4.39 4.39"><animate attributeName="stroke-dasharray" calcMode="spline" dur="5s" keySplines="0.5 0 0.5 1; 0.5 0 0.5 1" keyTimes="0; .5; 1" repeatCount="indefinite" values="3 6; 6 6; 3 6"/></path><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M20.48 20.48l-4.39-4.39"><animate attributeName="stroke-dasharray" calcMode="spline" dur="5s" keySplines="0.5 0 0.5 1; 0.5 0 0.5 1" keyTimes="0; .5; 1" repeatCount="indefinite" values="3 6; 6 6; 3 6"/></path><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M43.52 43.52l4.39 4.39"><animate attributeName="stroke-dasharray" calcMode="spline" dur="5s" keySplines="0.5 0 0.5 1; 0.5 0 0.5 1" keyTimes="0; .5; 1" repeatCount="indefinite" values="3 6; 6 6; 3 6"/></path><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M15.71 32H9.5"><animate attributeName="stroke-dasharray" calcMode="spline" dur="5s" keySplines="0.5 0 0.5 1; 0.5 0 0.5 1" keyTimes="0; .5; 1" repeatCount="indefinite" values="3 6; 6 6; 3 6"/></path><path fill="none" stroke="#f59e0b" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M48.29 32h6.21"><animate attributeName="stroke-dasharray" calcMode="spline" dur="5s" keySplines="0.5 0 0.5 1; 0.5 0 0.5 1" keyTimes="0; .5; 1" repeatCount="indefinite" values="3 6; 6 6; 3 6"/></path><animateTransform attributeName="transform" dur="45s" from="0 32 32" repeatCount="indefinite" to="360 32 32" type="rotate"/></g></svg>';
				case 'day-fog'        :
					return '<svg class="svg-icon svg-fog" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><path fill="none" stroke="#dddddd" stroke-linejoin="round" stroke-width="3" d="M46.5 31.5h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0h28a7 7 0 000-14z"/><g><path fill="none" stroke="#d1d5db" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M17 58h30"/><animateTransform attributeName="transform" begin="0s" dur="5s" repeatCount="indefinite" type="translate" values="-4 0; 4 0; -4 0"/></g><g><path fill="none" stroke="#d1d5db" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M17 52h30"/><animateTransform attributeName="transform" begin="-4s" dur="5s" repeatCount="indefinite" type="translate" values="-4 0; 4 0; -4 0"/></g></svg>';
				case 'rain'           :
					return '<svg class="svg-icon svg-rain" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><path fill="none" stroke="#dddddd" stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M43.67 45.5h2.83a7 7 0 000-14h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0"/><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M24.39 43.03l-.78 4.94"/><animateTransform attributeName="transform" dur="0.7s" repeatCount="indefinite" type="translate" values="1 -5; -2 10"/><animate attributeName="opacity" dur="0.7s" repeatCount="indefinite" values="0;1;1;0"/></g><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M31.39 43.03l-.78 4.94"/><animateTransform attributeName="transform" begin="-0.4s" dur="0.7s" repeatCount="indefinite" type="translate" values="1 -5; -2 10"/><animate attributeName="opacity" begin="-0.4s" dur="0.7s" repeatCount="indefinite" values="0;1;1;0"/></g><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M38.39 43.03l-.78 4.94"/><animateTransform attributeName="transform" begin="-0.2s" dur="0.7s" repeatCount="indefinite" type="translate" values="1 -5; -2 10"/><animate attributeName="opacity" begin="-0.2s" dur="0.7s" repeatCount="indefinite" values="0;1;1;0"/></g></svg>';
				case 'storm-showers'  :
					return '<svg class="svg-icon svg-storm-showers" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><path fill="none" stroke="#dddddd" stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M43.67 45.5h2.83a7 7 0 000-14h-.32a10.49 10.49 0 00-19.11-8 7 7 0 00-10.57 6 7.21 7.21 0 00.1 1.14A7.5 7.5 0 0018 45.5a4.19 4.19 0 00.5 0v0"/><g><path fill="#f59e0b" d="M30 36l-4 12h4l-2 10 10-14h-6l4-8h-6z"/><animate attributeName="opacity" dur="2s" repeatCount="indefinite" values="1;1;1;1;1;1;0.1;1;0.1;1;1;0.1;1;0.1;1"/></g></svg>';
				case 'windy'          :
					return '<svg class="svg-icon svg-windy" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><g><path fill="none" stroke="#888" stroke-linecap="round" stroke-miterlimit="10" stroke-width="3" d="M43.64 20a5 5 0 113.61 8.46h-35.5M29.14 44a5 5 0 103.61-8.46h-21"/><animateTransform attributeName="transform" dur="2s" repeatCount="indefinite" type="translate" values="-8 2; 0 -2; 8 0; 0 1; -8 2"/></g></svg>';
				case 'high'          :
					return '<svg class="svg-icon svg-high" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><g><path fill="none" stroke="#ef4444" stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M32 44V20l-5.79 6.89L32 20l5.78 6.89"/><animateTransform attributeName="transform" begin="0s" dur="1.5s" keyTimes="0.0; 0.5; 0.9; 1.0" repeatCount="indefinite" type="translate" values="0 0; 0 0; 0 -9; 0 -9"/><animate attributeName="opacity" dur="1.5s" keyTimes="0.0; 0.3; 0.8; 0.9; 1.0" repeatCount="indefinite" values="0; 1; 1; 0; 0"/></g></svg>';
				case 'low'            :
					return '<svg class="svg-icon svg-low" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><g><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M32 20v24l-5.79-6.89L32 44l5.78-6.89"/><animateTransform attributeName="transform" begin="0s" dur="1.5s" keyTimes="0.0; 0.5; 0.9; 1.0" repeatCount="indefinite" type="translate" values="0 0; 0 0; 0 9; 0 9"/><animate attributeName="opacity" dur="1.5s" keyTimes="0.0; 0.3; 0.8; 0.9; 1.0" repeatCount="indefinite" values="0; 1; 1; 0; 0"/></g></svg>';
				case 'raindrop'       :
					return '<svg class="svg-icon svg-raindrop" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><path fill="none" stroke="#2885c7" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M32 17c-6.09 9-10 14.62-10 20.09a10 10 0 0020 0C42 31.62 38.09 26 32 17z"/></svg>';
				case 'thermometer'    :
					return '<svg class="svg-icon svg-thermometer" aria-hidden="true" role="img" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><circle cx="32" cy="42" r="4" fill="#ef4444"/><path fill="none" stroke="#ef4444" stroke-linecap="round" stroke-miterlimit="10" stroke-width="2" d="M32 28.5v13"><animate attributeName="y1" dur="5s" repeatCount="indefinite" values="28.5;25.5;28.5"/></path><path fill="none" stroke="#bbbbbb" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M36 36.07v-17a4 4 0 10-8 0v17a7.12 7.12 0 00-3 5.83 7 7 0 1014 0 7.12 7.12 0 00-3-5.83zM32.5 25h3M32.5 21h3M32.5 29h3"/></svg>';
			}
		}
	}
}